package com.vaadin.copilot;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.exception.SuggestRestartException;
import com.vaadin.copilot.exception.report.ExceptionReport;
import com.vaadin.copilot.exception.report.ExceptionReportCreator;
import com.vaadin.copilot.exception.report.ExceptionReportRelevantPairData;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ErrorHandler extends CopilotCommand {

    private static final Logger logger = LoggerFactory.getLogger(ErrorHandler.class);

    public static final String EXCEPTION_MESSAGE_KEY = "exceptionMessage";
    public static final String EXCEPTION_STACKTRACE_KEY = "exceptionStacktrace";
    private static final String DETAILS_JSON_KEY = "details";
    private static final String EXCEPTION_REPORT_KEY = "exceptionReport";
    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    public static ArrayNode toJson(Throwable e) {
        StringWriter sw = new StringWriter();
        try (PrintWriter pw = new PrintWriter(sw)) {
            e.printStackTrace(pw);
        }
        // We don't need the stack trace after
        // "com.vaadin.base.devserver.DebugWindowConnection.onMessage" as that
        // is outside Copilot
        // But we want all "causes" parts of the stack trace
        String[] stack = sw.toString().split("\n");

        AtomicBoolean include = new AtomicBoolean(true);
        Stream<String> filtered = Arrays.stream(stack).filter(s -> {
            if (s.contains("com.vaadin.base.devserver.DebugWindowConnection.onMessage")) {
                include.set(false);
            }
            if (!s.startsWith("\t")) {
                include.set(true);
            }
            return include.get();
        });
        return filtered.map(TextNode::valueOf).collect(JacksonUtils.asArray());
    }

    public static void setErrorMessage(JsonNode respData, String error) {
        setError(respData, error, null);
    }

    public static void setError(JsonNode respData, String errorMessage, Throwable e) {
        setError(respData, errorMessage, e, null);
    }

    public static void setError(JsonNode respData, String errorMessage, Throwable e,
            ExceptionReportCreator exceptionReportCreator) {
        ObjectNode error = JacksonUtils.createObjectNode();
        error.put("message", errorMessage);

        if (e != null) {
            String exceptionMessage = e.getMessage();
            if (exceptionMessage == null) {
                exceptionMessage = "";
            }
            error.put(EXCEPTION_MESSAGE_KEY, exceptionMessage);
            error.set(EXCEPTION_STACKTRACE_KEY, ErrorHandler.toJson(e));
            if (e instanceof SuggestRestartException) {
                error.put("suggestRestart", true);
            }
            if (Copilot.isDevelopmentMode()) {
                logger.error("Stack trace because in Copilot development mode", e);
            }
        }
        if (exceptionReportCreator != null) {
            ExceptionReport report = exceptionReportCreator.create();

            error.set(EXCEPTION_REPORT_KEY, JacksonUtils.writeValue(report));
        }
        if (respData.isObject()) {
            ((ObjectNode) respData).set("error", error);
        }
    }

    /**
     * Sends an error message to frontend using dev tools. This method calls
     * {@link #sendErrorResponse(DevToolsInterface, String, JsonNode, String, Throwable, ExceptionReportCreator)}
     * with {@code null} ExceptionReportCreator object
     *
     * @param devToolsInterface
     *            to send error object
     * @param command
     *            that has failed
     * @param responseData
     *            Response data to fill
     * @param error
     *            Error message
     * @param e
     *            Exception to get stack trace
     */
    public static void sendErrorResponse(DevToolsInterface devToolsInterface, String command, JsonNode responseData,
            String error, Throwable e) {
        ExceptionReportCreator exceptionReportCreator = new ExceptionReportCreator();
        exceptionReportCreator.setTitle(error);
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Command", command));
        sendErrorResponse(devToolsInterface, command, responseData, error, e, exceptionReportCreator);
    }

    /**
     * Constructs an error that will be sent to client via
     * {@link DevToolsInterface}. Message is constructed by
     * {@link #setError(JsonNode, String, Throwable, ExceptionReportCreator)}
     *
     * @param devToolsInterface
     *            to send error object
     * @param command
     *            that has failed
     * @param responseData
     *            response data that is sent to client
     * @param error
     *            error message
     * @param e
     *            exception to get stack trace
     * @param exceptionReportCreator
     *            report creator to make exceptions reportable to GH. Might be null.
     */
    public static void sendErrorResponse(DevToolsInterface devToolsInterface, String command, JsonNode responseData,
            String error, Throwable e, ExceptionReportCreator exceptionReportCreator) {
        ErrorHandler.setError(responseData, error, e, exceptionReportCreator);
        devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
    }

    public ErrorHandler() {
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if ("error".equals(command)) {
            if (!allowedToSendErrors()) {
                return true;
            }

            String proKey = LocalProKey.get().getProKey();
            String machineId = MachineId.get();
            URI uri = copilotServerClient.getQueryURI("errors");
            String details = data.hasNonNull(DETAILS_JSON_KEY) ? data.get(DETAILS_JSON_KEY).asText() : "";
            ErrorsRequest trackingRequest = new ErrorsRequest(machineId, proKey, data.get("message").asText(),
                    Map.of(DETAILS_JSON_KEY, details, "versions", data.get("versions").asText()));
            HttpRequest request = copilotServerClient.buildRequest(uri, trackingRequest);
            copilotServerClient.getHttpClient().sendAsync(request, HttpResponse.BodyHandlers.ofString());
            AnalyticsClient.getInstance().track("error", Map.of("message", data.get("message").asText()));
            return true;
        }
        return false;
    }

    private boolean allowedToSendErrors() {
        if (!AnalyticsClient.isEnabled()) {
            return false;
        }

        UserInfo userInfo = UserInfoServerClient.getUserInfoWithLocalProKey();
        if (userInfo == null) {
            return false;
        } else if (userInfo.copilotProjectCannotLeaveLocalhost()) {
            return false;
        }
        return MachineConfiguration.get().isSendErrorReportsAllowed();
    }
}