package com.vaadin.copilot;

import java.io.File;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.vaadin.flow.component.UI;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.util.SharedUtil;

import elemental.json.JsonObject;

import org.apache.commons.lang3.StringUtils;

public class Util {

    private static final Pattern endsWithNumber = Pattern.compile("([0-9]+)$");

    /**
     * Takes a filename and increases a trailing number on the name (excluding the
     * extension).
     *
     * <p>
     * For example, "file1.txt" would return "file2.txt", "file.txt" would return
     * "file1.txt" and "file123.txt" would return "file124.txt". If the filename
     * doesn't end with a number, a 1 is added at the end.
     *
     * @param filename
     *            the string to increase
     * @return the input string with the trailing number increased by one, or with a
     *         1 added at the end if it didn't end with a number
     */
    public static String increaseTrailingNumber(String filename) {
        String[] parts = filename.split("\\.", 2);
        String base = parts[0];
        String extension = parts[1];

        Matcher matcher = endsWithNumber.matcher(base);
        if (matcher.find()) {
            return base.substring(0, matcher.start()) + (Integer.parseInt(matcher.group()) + 1) + "." + extension;
        }
        return base + "1." + extension;
    }

    /**
     * Finds the first folder inside the given folder that contains more than one
     * sub-folder.
     *
     * <p>
     *
     * @param javaSourceFolder
     *            the root java source folder
     * @return the first folder that contains more than one sub-folder
     */
    public static File getSinglePackage(File javaSourceFolder) {
        File[] files = javaSourceFolder.listFiles();
        if (files == null || files.length == 0 || files.length > 1) {
            // Empty folder or folder containing multiple files
            return javaSourceFolder;
        }

        if (files[0].isDirectory()) {
            return getSinglePackage(files[0]);
        }
        return javaSourceFolder;
    }

    /**
     * Converts a string to title case.
     *
     * <p>
     * For example, "hello world" becomes "Hello World".
     *
     * @param input
     *            the input string
     * @return the input string converted to title case
     */
    public static String titleCase(String input) {
        return Arrays.stream(input.split(" ")).map(SharedUtil::capitalize).collect(Collectors.joining(" "));
    }

    /**
     * Replaces any folder with the given name in the path with a new folder name.
     *
     * @param path
     *            the path to modify
     * @param oldFolderName
     *            the name of the folder to replace
     * @param newFolderName
     *            the name of the folder to replace with
     * @param parentFolder
     *            the name of the folder containing the folder to replace
     * @return the modified path
     */
    public static Path replaceFolderInPath(Path path, String oldFolderName, String newFolderName, String parentFolder) {

        Path newPath = path.isAbsolute() ? Path.of("/") : Path.of("");

        boolean inParent = false;

        // Iterate over the path components
        for (Path part : path) {
            // If the current part matches the old folder name, replace it
            if (inParent && part.toString().equals(oldFolderName)) {
                newPath = newPath.resolve(newFolderName);
            } else {
                newPath = newPath.resolve(part);
            }
            inParent = parentFolder.equals(part.toString());
        }
        return newPath;
    }

    /**
     * Finds the common ancestor path for the given paths.
     * <p>
     * For instance if you pass in /foo/bar/baz and /foo/bar, this returns /foo/bar
     *
     * @param paths
     *            the paths to process
     * @return the common ancestor path
     */
    public static Optional<Path> findCommonAncestor(List<Path> paths) {
        if (paths == null || paths.isEmpty()) {
            return Optional.empty();
        }
        Path commonPath = paths.get(0);
        for (int i = 1; i < paths.size(); i++) {
            Path otherPath = paths.get(i);
            commonPath = Path.of(StringUtils.getCommonPrefix(commonPath.toString(), otherPath.toString()));
        }
        return Optional.of(commonPath);
    }

    /**
     * Finds a file related to the current view, to be able to determine which
     * folder / project module to create new files in.
     *
     * @param projectManager
     *            the project manager
     * @param currentView
     *            JSON data for the current view, with either "viewFile" pointing to
     *            a Hilla view file or "uiId" refering to an open Flow UI
     * @return a file (Java or TSX) used for the current view
     */
    public static Optional<File> findCurrentViewFile(ProjectManager projectManager, JsonObject currentView) {
        if (currentView == null) {
            return Optional.empty();
        }
        if (currentView.hasKey("views") && currentView.getArray("views").length() > 0) {
            return Optional.of(new File(currentView.getArray("views").getString(0)));
        }
        if (currentView.hasKey("uiId")) {
            int uiId = (int) currentView.getNumber("uiId");
            VaadinSession session = projectManager.getVaadinSession();
            session.lock();
            try {
                UI ui = session.getUIById(uiId);
                return Optional.of(
                        new ComponentSourceFinder(projectManager)._getSourceLocation(ui.getCurrentView()).javaFile());
            } finally {
                session.unlock();
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the module where the (first of) currently open view(s) is defined.
     * 
     * @param projectManager
     *            the project manager
     * @param currentView
     *            JSON data for the current view, with either "viewFile" pointing to
     *            a Hilla view file or "uiId" referring to an open Flow UI
     * @return the module where the current view is defined
     */
    public static Optional<JavaSourcePathDetector.ModuleInfo> findCurrentModule(ProjectManager projectManager,
            JsonObject currentView) {
        return findCurrentViewFile(projectManager, currentView).flatMap(ProjectFileManager.getInstance()::findModule);
    }
}
