package com.vaadin.copilot.plugins.themeeditor;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.IntStream;

import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.plugins.themeeditor.utils.LineNumberVisitor;
import com.vaadin.copilot.plugins.themeeditor.utils.LocalClassNameVisitor;
import com.vaadin.copilot.plugins.themeeditor.utils.LocalClassNamesVisitor;
import com.vaadin.copilot.plugins.themeeditor.utils.ThemeEditorException;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.comments.Comment;
import com.github.javaparser.ast.comments.LineComment;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.SimpleName;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithBlockStmt;
import com.github.javaparser.ast.nodeTypes.NodeWithExpression;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.Statement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JavaSourceModifier extends CopilotEditor {

    private static final String LOCAL_CLASSNAME_COMMENT = "<theme-editor-local-classname>";

    private static final String CANNOT_GENERATE_METADATA = "Cannot generate metadata.";

    public static boolean hasLocalClassnameComment(Node n) {
        return n.getComment().map(Comment::getContent).map(String::trim)
                .filter(JavaSourceModifier.LOCAL_CLASSNAME_COMMENT::equals).isPresent();
    }

    public JavaSourceModifier(ProjectManager projectManager) {
        super(projectManager);
    }

    /**
     * Adds local component class name if not already present, updates value
     * otherwise.
     *
     * @param uiId
     *            uiId of target component's UI
     * @param nodeId
     *            nodeIf of target component
     * @param className
     *            className to be set
     */
    public void setLocalClassName(Integer uiId, Integer nodeId, String className) {
        assert uiId != null && nodeId != null && className != null;
        ComponentTypeAndSourceLocation componentSource = sourceFinder.findTypeAndSourceLocation(uiId, nodeId);
        try {
            setLocalClassName(componentSource, className, false);
            if (hasOverlay(componentSource)) {
                setLocalClassName(componentSource, className, true);
            }
        } catch (Exception e) {
            throw new ThemeEditorException("Cannot set local classname.", e);
        }

    }

    protected void setLocalClassName(ComponentTypeAndSourceLocation componentSource, String className,
            boolean overlay) {
        try {
            File sourceFile = componentSource.javaFile();
            int sourceOffset = modifyClass(sourceFile, cu -> {
                SimpleName scope = findLocalVariableOrField(cu,
                        componentSource.getCreateLocationOrThrow().lineNumber());
                Node newNode = createAddClassNameStatement(scope, className, overlay);
                Modification mod;
                ExpressionStmt stmt = findLocalClassNameStmt(cu, componentSource, overlay);
                if (stmt == null) {
                    Node node = findNode(cu, componentSource);
                    Where where = findModificationWhere(cu, componentSource);
                    mod = switch (where) {
                    case AFTER -> Modification.insertLineAfter(node, newNode);
                    case INSIDE -> Modification.insertAtEndOfBlock(node, newNode);
                    case BEFORE -> Modification.insertLineBefore(node, newNode);
                    };
                } else {
                    mod = Modification.replace(stmt, newNode);
                }
                return Collections.singletonList(mod);
            });

            if (sourceOffset != 0) {
                ComponentTracker.refreshLocation(componentSource.getCreateLocationOrThrow(), sourceOffset);
            }

        } catch (UnsupportedOperationException ex) {
            throw new ThemeEditorException(ex);
        }
    }

    /**
     * Gets tag name of given component.
     *
     * @param uiId
     *            uiId of target component's UI
     * @param nodeId
     *            nodeIf of target component
     * @return tag name of given element
     */
    public String getTag(Integer uiId, Integer nodeId) {
        assert uiId != null && nodeId != null;
        ComponentTypeAndSourceLocation componentSource = sourceFinder.findTypeAndSourceLocation(uiId, nodeId);
        try {
            return componentSource.component().getElement().getTag();
        } catch (Exception e) {
            throw new ThemeEditorException("Cannot get tag of component.", e);
        }
    }

    /**
     * Gets component local classname if exists.
     *
     * @param uiId
     *            uiId of target component's UI
     * @param nodeId
     *            nodeIf of target component
     * @return component local classname
     */
    public String getLocalClassName(Integer uiId, Integer nodeId) {
        assert uiId != null && nodeId != null;
        try {
            ComponentTypeAndSourceLocation componentSource = sourceFinder.findTypeAndSourceLocation(uiId, nodeId);
            CompilationUnit cu = getCompilationUnit(componentSource);
            ExpressionStmt localClassNameStmt = findLocalClassNameStmt(cu, componentSource, false);
            if (localClassNameStmt != null) {
                return localClassNameStmt.getExpression().asMethodCallExpr().getArgument(0).asStringLiteralExpr()
                        .asString();
            }

            return null;
        } catch (IOException e) {
            throw new ThemeEditorException("Cannot get local classname.", e);
        }
    }

    /**
     * Removes local class name of given component.
     *
     * @param uiId
     *            uiId of target component's UI
     * @param nodeId
     *            nodeIf of target component
     */
    public void removeLocalClassName(Integer uiId, Integer nodeId) {
        assert uiId != null && nodeId != null;

        ComponentTypeAndSourceLocation componentSource = sourceFinder.findTypeAndSourceLocation(uiId, nodeId);
        try {
            removeLocalClassName(componentSource, false);
            if (hasOverlay(componentSource)) {
                removeLocalClassName(componentSource, true);
            }
        } catch (Exception e) {
            throw new ThemeEditorException("Cannot remove local classname.", e);
        }
    }

    public void removeLocalClassName(ComponentTypeAndSourceLocation componentSource, boolean overlay) {
        int sourceOffset = modifyClass(componentSource.javaFile(), cu -> {
            ExpressionStmt localClassNameStmt = findLocalClassNameStmt(cu, componentSource, overlay);
            if (localClassNameStmt != null) {
                return Collections.singletonList(Modification.remove(localClassNameStmt));
            }
            throw new ThemeEditorException("Local classname not present.");
        });

        if (sourceOffset != 0) {
            ComponentTracker.refreshLocation(componentSource.getCreateLocationOrThrow(), sourceOffset);
        }
    }

    /**
     * Checks if component can be accessed within source code.
     *
     * @param uiId
     *            uiId of target component's UI
     * @param nodeId
     *            nodeIf of target component
     * @return true if component is accessible, false otherwise
     */
    public boolean isAccessible(Integer uiId, Integer nodeId) {
        assert uiId != null && nodeId != null;

        ComponentTypeAndSourceLocation componentSource = sourceFinder.findTypeAndSourceLocation(uiId, nodeId);

        try {
            CompilationUnit cu = getCompilationUnit(componentSource);
            findModificationWhere(cu, componentSource);
            return true;
        } catch (Exception ex) {
            getLogger().debug(ex.getMessage(), ex);
        }
        return false;
    }

    /**
     * Creates suggested local classname based on component tag.
     *
     * @param uiId
     *            uiId of target component's UI
     * @param nodeId
     *            nodeIf of target component
     * @return suggested local classname
     */
    public String getSuggestedClassName(Integer uiId, Integer nodeId) {
        assert uiId != null && nodeId != null;
        try {
            ComponentTypeAndSourceLocation componentSource = sourceFinder.findTypeAndSourceLocation(uiId, nodeId);
            ComponentTracker.Location createLocation = componentSource.getCreateLocationOrThrow();
            String fileName = SharedUtil.upperCamelCaseToDashSeparatedLowerCase(
                    createLocation.filename().substring(0, createLocation.filename().indexOf(".")));
            String tagName = componentSource.component().getElement().getTag().replace("vaadin-", "");

            CompilationUnit cu = getCompilationUnit(componentSource);
            LocalClassNamesVisitor visitor = new LocalClassNamesVisitor();
            cu.accept(visitor, null);
            List<String> existingClassNames = visitor.getArguments();
            String suggestion = fileName + "-" + tagName + "-";
            // suggest classname "filename-tagname-" + (1 : 99)
            return IntStream.range(1, 100).mapToObj(i -> suggestion + i).filter(i -> !existingClassNames.contains(i))
                    .findFirst().orElse(null);
        } catch (Exception e) {
            throw new ThemeEditorException(CANNOT_GENERATE_METADATA, e);
        }
    }

    protected Statement createAddClassNameStatement(SimpleName scope, String className, boolean overlay) {
        MethodCallExpr methodCallExpr = new MethodCallExpr(overlay ? "setOverlayClassName" : "addClassName");
        if (scope != null) {
            methodCallExpr.setScope(new NameExpr(scope));
        }
        methodCallExpr.getArguments().add(new StringLiteralExpr(className));
        Statement statement = new ExpressionStmt(methodCallExpr);
        statement.setComment(new LineComment(LOCAL_CLASSNAME_COMMENT));
        return statement;
    }

    protected Component getComponent(VaadinSession session, int uiId, int nodeId) {
        Element element = session.findElement(uiId, nodeId);
        Optional<Component> c = element.getComponent();
        if (c.isEmpty()) {
            throw new ThemeEditorException(
                    "Only component locations are tracked. The given node id refers to an element and not a component.");
        }
        return c.get();
    }

    protected CompilationUnit getCompilationUnit(ComponentTypeAndSourceLocation componentSource) throws IOException {
        var source = projectManager.readFile(componentSource.javaFile());
        return new JavaRewriter(source).getCompilationUnit();
    }

    protected ExpressionStmt findLocalClassNameStmt(CompilationUnit cu, ComponentTypeAndSourceLocation componentSource,
            boolean overlay) {
        SimpleName scope = findLocalVariableOrField(cu, componentSource.getCreateLocationOrThrow().lineNumber());
        Node parentBlockNode = findParentBlockNode(cu, componentSource.component());
        return parentBlockNode.accept(new LocalClassNameVisitor(overlay), scope != null ? scope.getIdentifier() : null);
    }

    protected Node findParentBlockNode(CompilationUnit cu, Component component) {
        ComponentTracker.Location createLocation = sourceFinder.findTypeAndSourceLocation(component, false)
                .getCreateLocationOrThrow();
        Node node = cu.accept(new LineNumberVisitor(), createLocation.lineNumber());
        if (node instanceof BlockStmt) {
            return node;
        }
        while (node.getParentNode().isPresent()) {
            node = node.getParentNode().get();
            if (node instanceof BlockStmt blockStmt) {
                return blockStmt;
            }
        }
        // fallback to CU
        return cu;
    }

    protected Where findModificationWhere(CompilationUnit cu, ComponentTypeAndSourceLocation componentSource) {
        Node node = findNode(cu, componentSource);
        if (node instanceof NodeWithBlockStmt<?>) {
            return Where.INSIDE;
        }
        if (node instanceof NodeWithExpression<?> expr
                && (expr.getExpression().isAssignExpr() || expr.getExpression().isVariableDeclarationExpr())) {
            return Where.AFTER;
        }
        throw new ThemeEditorException("Cannot apply classname for " + node);
    }

    protected Node findNode(CompilationUnit cu, ComponentTypeAndSourceLocation componentSource) {
        Node node = cu.accept(new LineNumberVisitor(), componentSource.getCreateLocationOrThrow().lineNumber());
        if (node == null) {
            throw new ThemeEditorException("Cannot find component.");
        }
        return node;
    }

    protected boolean hasOverlay(ComponentTypeAndSourceLocation componentSource) {
        try {
            // HasOverlayClassName interface is not part of flow-server
            componentSource.component().getClass().getMethod("setOverlayClassName", String.class);
            return true;
        } catch (NoSuchMethodException e) {
            return false;
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaSourceModifier.class);
    }
}
