package com.vaadin.copilot;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.ServerInfo;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.startup.ApplicationConfiguration;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.ProKey;
import elemental.json.Json;
import elemental.json.JsonObject;
import elemental.json.JsonValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class AICommandHandler implements CopilotCommand {
    private static final String MESSAGE_PROMPT_FAILED = "prompt-failed";
    private static final String MESSAGE_PROMPT_OK = "prompt-ok";
    public static final String SERVER_URL_ENV = "copilot.serverBaseUrl";
    private final HttpClient httpClient;
    private final ObjectMapper objectMapper;
    private Map<String, String> metadata;
    private final ProjectManager projectManager;

    public AICommandHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_2)
                .followRedirects(HttpClient.Redirect.NORMAL).build();
        objectMapper = new ObjectMapper();
        metadata = new HashMap<>();
        ServerInfo serverInfo = new ServerInfo();
        metadata = serverInfo.getVersions().stream()
                .collect(Collectors.toMap(ServerInfo.NameAndVersion::name,
                        ServerInfo.NameAndVersion::version));
    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (command.equals("prompt-text")) {
            String prompt = data.getString("text");

            // Hilla source files;
            Set<String> filenames = JsonUtils.stream(data.getArray("sources"))
                    .map(JsonValue::asString).collect(Collectors.toSet());
            getLogger().debug("Hilla fileNames: {}", filenames);

            Map<String, String> sources = new HashMap<>();
            for (String filename : filenames) {
                try {
                    sources.put(filename, projectManager.readFile(filename));
                } catch (IOException e) {
                    getLogger().error(
                            "Error reading requested project Hilla/React file: "
                                    + filename,
                            e);
                    devToolsInterface.send(
                            Copilot.PREFIX + MESSAGE_PROMPT_FAILED,
                            Json.createObject());
                    return true;
                }
            }

            // Java source files:
            if (data.hasKey("uiid")) {
                Number uiId = data.getNumber("uiid");
                Map<String, String> fileSourceMapJava;
                VaadinSession session = VaadinSession.getCurrent();
                try {
                    session.lock();
                    fileSourceMapJava = getJavaSourceMap(uiId, session);
                } catch (IOException e) {
                    getLogger().error(
                            "Error reading requested project Flow Java files",
                            e);
                    devToolsInterface.send(
                            Copilot.PREFIX + MESSAGE_PROMPT_FAILED,
                            Json.createObject());
                    return true;
                } finally {
                    session.unlock();
                }
                sources.putAll(fileSourceMapJava);
            }

            String proKey = Optional.ofNullable(LocalProKey.get())
                    .map(ProKey::getProKey).orElse(null);
            try {
                CopilotServerRequest req = new CopilotServerRequest(prompt,
                        sources, metadata, proKey);
                queryCopilotServer(req, response -> {
                    try {
                        handleQueryResponse(response);
                        devToolsInterface.send(
                                Copilot.PREFIX + MESSAGE_PROMPT_OK,
                                Json.createObject());
                    } catch (IOException e) {
                        getLogger().error(
                                "Error handling copilot server response", e);
                        devToolsInterface.send(
                                Copilot.PREFIX + MESSAGE_PROMPT_FAILED,
                                Json.createObject());
                    }
                });
            } catch (Exception e) {
                getLogger().error("Error querying copilot server", e);
                devToolsInterface.send(Copilot.PREFIX + MESSAGE_PROMPT_FAILED,
                        Json.createObject());
                return true;
            }
            return true;
        }
        return false;
    }

    private Map<String, String> getJavaSourceMap(Number uiId,
            VaadinSession session) throws IOException {

        UI ui = session.getUIById(uiId.intValue());
        List<Component> componentList = new ArrayList<>();
        Map<String, String> sources = new HashMap<>();

        addComponents(ui, componentList);

        Set<ComponentTracker.Location> locations = new HashSet<>();

        for (Component component : componentList) {
            ComponentTracker.Location create = ComponentTracker
                    .findCreate(component);
            ComponentTracker.Location attach = ComponentTracker
                    .findAttach(component);
            if (create != null && attach != null) {
                // e.g. the UI has no create location and we are not interested
                // in its attach
                locations.add(create);
                locations.add(attach);
            }
        }

        ApplicationConfiguration applicationConfiguration = ApplicationConfiguration
                .get(session.getService().getContext());
        for (ComponentTracker.Location location : locations) {
            File javaFile = location.findJavaFile(applicationConfiguration);
            if (!sources.containsKey(location.filename())) {
                // The component locator tracks the Java class and guesses the
                // Java file, so it's possible the Java file is not in the
                // project
                ArrayList<String> javaFileNames = new ArrayList<>();
                if (javaFile.exists()) {
                    javaFileNames.add(javaFile.getName());
                    sources.put(javaFile.getAbsolutePath(),
                            projectManager.readFile(javaFile.getPath()));
                }
                getLogger().debug("Java filenames: {}", javaFileNames);
            }
        }
        return sources;
    }

    private void addComponents(Component component,
            List<Component> componentList) {
        componentList.add(component);
        component.getChildren().forEach(c -> addComponents(c, componentList));
    }

    private void queryCopilotServer(CopilotServerRequest req,
            Consumer<CopilotServerResponse> responseHandler) {

        URI queryUri;
        try {
            queryUri = new URI(getServerBaseUrl() + "query");
        } catch (URISyntaxException e) {
            throw new IllegalStateException(
                    "Invalid server configuration, server uri is wrong", e);
        }
        String json;
        try {
            json = objectMapper.writeValueAsString(req);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid request", e);
        }
        if (isDevelopment()) {
            getLogger().info("Querying copilot server at {} using {}", queryUri,
                    json);
        }

        HttpRequest request = HttpRequest.newBuilder().uri(queryUri)
                .POST(HttpRequest.BodyPublishers.ofString(json))
                .header("Content-Type", "application/json")
                .timeout(Duration.ofSeconds(120)).build();

        Consumer<String> responseParser = (responseJson) -> {
            try {
                getLogger().info("Response: {}", responseJson);
                CopilotServerResponse response = objectMapper
                        .readValue(responseJson, CopilotServerResponse.class);
                responseHandler.accept(response);
            } catch (JsonProcessingException e) {
                throw new IllegalArgumentException(
                        "Unable to parse copilot server response", e);
            }
        };

        httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(HttpResponse::body).thenAccept(responseParser)
                .join();

    }

    private String getServerBaseUrl() {
        String serverBaseUrl = System.getenv(SERVER_URL_ENV);
        if (serverBaseUrl != null) {
            return serverBaseUrl;
        }

        if (isDevelopment()) {
            // Try a localhost server
            try {
                String localhostUrl = "http://localhost:8081/v1/";
                String statusUrl = localhostUrl + "actuator/health";

                HttpRequest request = HttpRequest.newBuilder()
                        .uri(new URI(statusUrl)).timeout(Duration.ofSeconds(1))
                        .build();
                HttpResponse<String> response = httpClient.send(request,
                        HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() == 200) {
                    return localhostUrl;
                }
            } catch (Exception e) {
                // Ignore
            }
            return "https://copilot.stg.vaadin.com/v1/";
        } else {
            return "https://copilot.vaadin.com/v1/";
        }
    }

    private static boolean isDevelopment() {
        return System.getProperty("copilot.development") != null;
    }

    private void handleQueryResponse(CopilotServerResponse response)
            throws IOException {
        if (response.code() == CopilotServerResponseCode.ERROR) {
            getLogger().error(
                    "Copilot server returned error because an internal error."
                            + " The reason could be a malformed request or a timeout.");
            return;
        } else if (response.code() == CopilotServerResponseCode.ERROR_REQUEST) {
            getLogger().error(
                    "Copilot server returned error because an internal error."
                            + " The reason could be a malformed request or a timeout.");
            return;
        } else if (response.code() == CopilotServerResponseCode.NOTHING) {
            getLogger().debug("Copilot server returned no changes");
            return;
        } else if (response.code() == CopilotServerResponseCode.HILLA_REACT) {
            getLogger().debug("Copilot server returned Hilla/React changes");
        } else if (response.code() == CopilotServerResponseCode.FLOW) {
            getLogger().debug("Copilot server returned Flow changes");
        }

        for (Map.Entry<String, String> change : response.changes().entrySet()) {
            try {
                projectManager.writeFile(change.getKey(), change.getValue());
            } catch (IOException e) {
                throw new IOException(
                        "Unable to write file (" + change.getKey()
                                + ") with data from copilot server response",
                        e);
            }
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

}
