/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.collaborationengine;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.vaadin.collaborationengine.Backend;
import com.vaadin.collaborationengine.BackendUtil;
import com.vaadin.collaborationengine.CollaborationEngine;
import com.vaadin.collaborationengine.CollaborationEngineConfiguration;
import com.vaadin.collaborationengine.FileLicenseStorage;
import com.vaadin.collaborationengine.JsonConversionException;
import com.vaadin.collaborationengine.JsonUtil;
import com.vaadin.collaborationengine.LicenseEvent;
import com.vaadin.collaborationengine.LicenseStorage;
import com.vaadin.flow.internal.MessageDigestUtil;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.YearMonth;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

class LicenseHandler {
    private static final String EVENT_LOG_NAME = LicenseHandler.class.getName();
    static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_DATE;
    static final ObjectMapper MAPPER = LicenseHandler.createObjectMapper();
    private final CollaborationEngine ce;
    private final CollaborationEngineConfiguration configuration;
    private final Backend backend;
    final LicenseStorage licenseStorage;
    final LicenseInfo license;
    final Backend.EventLog licenseEventLog;
    private UUID lastSnapshotId;
    private StatisticsInfo statisticsCache;
    private final List<UUID> backendNodes = new ArrayList<UUID>();
    private boolean leader;

    LicenseHandler(CollaborationEngine collaborationEngine) {
        this.ce = collaborationEngine;
        this.configuration = collaborationEngine.getConfiguration();
        this.backend = this.configuration.getBackend();
        if (this.configuration.isLicenseCheckingEnabled()) {
            LicenseStorage configuredStorage = this.configuration.getLicenseStorage();
            this.licenseStorage = configuredStorage != null ? configuredStorage : new FileLicenseStorage(this.configuration);
            String licenseProperty = this.configuration.getLicenseProperty();
            if (licenseProperty != null) {
                this.license = this.parseLicense(this.getLicenseFromProperty(licenseProperty));
            } else {
                Path dataDirPath = this.configuration.getDataDirPath();
                if (dataDirPath == null) {
                    throw FileLicenseStorage.createDataDirNotConfiguredException();
                }
                Path licenseFilePath = LicenseHandler.createLicenseFilePath(dataDirPath);
                this.license = this.parseLicense(this.getLicenseFromFile(licenseFilePath));
            }
            if (this.license.endDate.isBefore(this.getCurrentDate())) {
                CollaborationEngine.LOGGER.warn("Your Collaboration Engine license has expired. Your application will still continue to work, but the collaborative features will be disabled. Please contact Vaadin about obtaining a new, up-to-date license for your application. https://vaadin.com/collaboration");
            }
            this.licenseEventLog = this.backend.openEventLog(EVENT_LOG_NAME);
            this.backend.addMembershipListener(event -> {
                UUID nodeId = event.getNodeId();
                switch (event.getType()) {
                    case JOIN: {
                        this.handleNodeJoin(nodeId);
                        break;
                    }
                    case LEAVE: {
                        this.handleNodeLeave(nodeId);
                    }
                }
            });
            BackendUtil.initializeFromSnapshot(this.ce, this::initializeFromSnapshot).thenAccept(uuid -> {
                this.lastSnapshotId = uuid;
            });
        } else {
            this.licenseEventLog = null;
            this.licenseStorage = null;
            this.license = null;
        }
    }

    boolean isLeader() {
        return this.leader;
    }

    private void becomeLeader() {
        this.leader = true;
    }

    private void handleNodeJoin(UUID nodeId) {
        if (this.backendNodes.isEmpty() && this.backend.getNodeId().equals(nodeId)) {
            this.becomeLeader();
        }
        this.backendNodes.add(nodeId);
    }

    private void handleNodeLeave(UUID nodeId) {
        this.backendNodes.remove(nodeId);
        if (!this.backendNodes.isEmpty() && this.backendNodes.get(0).equals(this.backend.getNodeId())) {
            this.becomeLeader();
        }
    }

    private CompletableFuture<UUID> initializeFromSnapshot() {
        return this.backend.loadLatestSnapshot(EVENT_LOG_NAME).thenCompose(this::loadAndSubscribe);
    }

    private CompletableFuture<UUID> loadAndSubscribe(Backend.Snapshot snapshot) {
        CompletableFuture<UUID> future = new CompletableFuture<UUID>();
        try {
            UUID latestChange = null;
            if (snapshot != null) {
                latestChange = this.loadSnapshot(JsonUtil.fromString(snapshot.getPayload())).getLatestChange();
                this.licenseEventLog.subscribe(latestChange, this::handleChangeEvent);
            } else {
                this.loadFromStorage();
                this.licenseEventLog.subscribe(null, this::handleChangeEvent);
            }
            future.complete(latestChange);
        }
        catch (Backend.EventIdNotFoundException e) {
            future.completeExceptionally(e);
        }
        return future;
    }

    private Snapshot loadSnapshot(ObjectNode node) {
        try {
            Snapshot snapshot = (Snapshot)MAPPER.treeToValue((TreeNode)node, Snapshot.class);
            this.statisticsCache = snapshot.getStatistics();
            return snapshot;
        }
        catch (JsonProcessingException | IllegalArgumentException e) {
            throw new IllegalStateException("Collaboration Engine failed to load license usage data.", e);
        }
    }

    private void loadFromStorage() {
        this.statisticsCache = new StatisticsInfo(this.license.key, Collections.emptyMap(), Collections.emptyMap());
        YearMonth month = YearMonth.from(this.getCurrentDate());
        this.licenseStorage.getUserEntries(this.license.key, month).forEach(userId -> this.statisticsCache.addUserEntry(month, (String)userId));
        this.licenseStorage.getLatestLicenseEvents(this.license.key).forEach(this.statisticsCache::setLicenseEvent);
    }

    private void handleChangeEvent(UUID eventId, String payload) {
        ObjectNode event = JsonUtil.fromString(payload);
        String changeType = event.get("type").asText();
        String licenseKey = event.get("license-key").asText();
        if ("license-user".equals(changeType)) {
            YearMonth month = YearMonth.parse(event.get("year-month").asText());
            String userId = event.get("user-id").asText();
            this.statisticsCache.addUserEntry(month, userId);
            if (this.leader) {
                this.licenseStorage.addUserEntry(licenseKey, month, userId);
            }
        } else if ("license-event".equals(changeType)) {
            String eventName = event.get("event-name").asText();
            LocalDate latestOccurrence = LocalDate.parse(event.get("event-occurrence").asText());
            this.statisticsCache.setLicenseEvent(eventName, latestOccurrence);
            if (this.leader) {
                this.notifyLicenseEventHandler(eventName);
                this.licenseStorage.setLicenseEvent(licenseKey, eventName, latestOccurrence);
            }
        }
        if (this.lastSnapshotId == null) {
            Snapshot snapshot = new Snapshot(eventId, this.statisticsCache);
            try {
                this.backend.replaceSnapshot(EVENT_LOG_NAME, null, UUID.randomUUID(), MAPPER.writeValueAsString((Object)snapshot));
                this.backend.loadLatestSnapshot(EVENT_LOG_NAME).thenAccept(s -> {
                    this.lastSnapshotId = s.getId();
                });
            }
            catch (JsonProcessingException e) {
                throw new JsonConversionException("Cannot serialize snapshot", e);
            }
        }
    }

    private void notifyLicenseEventHandler(String eventName) {
        LicenseEvent.LicenseEventType type = LicenseEvent.LicenseEventType.valueOf(eventName);
        this.configuration.getLicenseEventHandler().handleLicenseEvent(new LicenseEvent(this.ce, type, switch (type) {
            case LicenseEvent.LicenseEventType.GRACE_PERIOD_STARTED -> {
                LocalDate gracePeriodEnd = this.getCurrentDate().plusDays(31L);
                yield type.createMessage(gracePeriodEnd.format(DATE_FORMATTER));
            }
            case LicenseEvent.LicenseEventType.LICENSE_EXPIRES_SOON -> type.createMessage(this.license.endDate.format(DATE_FORMATTER));
            default -> type.createMessage(new Object[0]);
        }));
    }

    private Reader getLicenseFromProperty(String licenseProperty) {
        byte[] license = Base64.getDecoder().decode(licenseProperty);
        return new InputStreamReader(new ByteArrayInputStream(license));
    }

    private Reader getLicenseFromFile(Path licenseFilePath) {
        try {
            return Files.newBufferedReader(licenseFilePath);
        }
        catch (NoSuchFileException e) {
            throw this.createLicenseNotFoundException(licenseFilePath, e);
        }
        catch (IOException e) {
            throw new IllegalStateException("Collaboration Engine wasn't able to read the license file at '" + licenseFilePath + "'. Check that the file is readable by the app, and not locked.", e);
        }
    }

    private LicenseInfo parseLicense(Reader licenseReader) {
        try {
            JsonNode licenseJson = MAPPER.readTree(licenseReader);
            LicenseInfoWrapper licenseInfoWrapper = (LicenseInfoWrapper)MAPPER.treeToValue((TreeNode)licenseJson, LicenseInfoWrapper.class);
            String calculatedChecksum = LicenseHandler.calculateChecksum(licenseJson.get("content"));
            if (licenseInfoWrapper.checksum == null || !licenseInfoWrapper.checksum.equals(calculatedChecksum)) {
                throw this.createLicenseInvalidException(null);
            }
            return licenseInfoWrapper.content;
        }
        catch (IOException e) {
            throw this.createLicenseInvalidException(e);
        }
    }

    synchronized boolean registerUser(String userId) {
        boolean hasActiveSeat;
        LocalDate currentDate = this.getCurrentDate();
        if (this.isGracePeriodEnded(currentDate)) {
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.GRACE_PERIOD_ENDED);
        }
        if (this.license.endDate.isBefore(currentDate)) {
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.LICENSE_EXPIRED);
            return false;
        }
        if (this.license.endDate.minusDays(31L).isBefore(currentDate)) {
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.LICENSE_EXPIRES_SOON);
        }
        YearMonth month = YearMonth.from(currentDate);
        List<String> users = this.statisticsCache.getUserEntries(month);
        int effectiveQuota = this.isGracePeriodOngoing(currentDate) ? this.license.quota * 10 : this.license.quota;
        boolean bl = hasActiveSeat = users.size() <= effectiveQuota ? users.contains(userId) : users.stream().limit(effectiveQuota).anyMatch(user -> user.equals(userId));
        if (hasActiveSeat) {
            return true;
        }
        if (users.size() >= effectiveQuota) {
            if (this.getGracePeriodStarted() != null) {
                return false;
            }
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.GRACE_PERIOD_STARTED);
        }
        ObjectNode entry = JsonUtil.createUserEntry(this.license.key, month, userId);
        this.licenseEventLog.submitEvent(UUID.randomUUID(), JsonUtil.toString(entry));
        return true;
    }

    LocalDate getGracePeriodStarted() {
        return this.statisticsCache.getLatestLicenseEvents().get(LicenseEvent.LicenseEventType.GRACE_PERIOD_STARTED.name());
    }

    private boolean isGracePeriodOngoing(LocalDate currentDate) {
        return this.getGracePeriodStarted() != null && !this.isGracePeriodEnded(currentDate);
    }

    private boolean isGracePeriodEnded(LocalDate currentDate) {
        return this.getGracePeriodStarted() != null && currentDate.isAfter(this.getLastGracePeriodDate());
    }

    private LocalDate getLastGracePeriodDate() {
        return this.getGracePeriodStarted().plusDays(30L);
    }

    private void fireLicenseEvent(LicenseEvent.LicenseEventType type) {
        String eventName = type.name();
        if (this.statisticsCache.getLatestLicenseEvents().get(eventName) != null) {
            return;
        }
        ObjectNode event = JsonUtil.createLicenseEvent(this.license.key, eventName, this.getCurrentDate());
        this.licenseEventLog.submitEvent(UUID.randomUUID(), JsonUtil.toString(event));
    }

    private LocalDate getCurrentDate() {
        return LocalDate.now(this.ce.getClock());
    }

    private RuntimeException createLicenseInvalidException(Throwable cause) {
        return new IllegalStateException("The content of the license property or file is not valid. If you have made any changes to the file, please revert those changes. If that's not possible, contact Vaadin to get a new copy of the license file.", cause);
    }

    private RuntimeException createLicenseNotFoundException(Path licenseFilePath, Throwable cause) {
        return new IllegalStateException("Collaboration Engine failed to find the license file at '" + licenseFilePath + ". Using Collaboration Engine in production requires a valid license property or file. Instructions for obtaining a license can be found in the Vaadin documentation. If you already have a license, make sure that the 'vaadin.ce.license' property is set or, if you have a license file, the 'vaadin.ce.dataDir' property is pointing to the correct directory and that the directory contains the license file.", cause);
    }

    Map<YearMonth, Set<String>> getStatistics() {
        return this.statisticsCache.copyMap(this.statisticsCache.statistics);
    }

    static String calculateChecksum(JsonNode node) throws JsonProcessingException {
        return Base64.getEncoder().encodeToString(MessageDigestUtil.sha256((String)MAPPER.writeValueAsString((Object)node)));
    }

    static ObjectMapper createObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule((Module)new JavaTimeModule());
        objectMapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.NON_PRIVATE);
        objectMapper.setDateFormat((DateFormat)new SimpleDateFormat("yyyy-MM-dd"));
        return objectMapper;
    }

    static Path createLicenseFilePath(Path dirPath) {
        return Paths.get(dirPath.toString(), "ce-license.json");
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    static class LicenseInfo {
        final String key;
        final int quota;
        final LocalDate endDate;

        @JsonCreator
        LicenseInfo(@JsonProperty(value="key", required=true) String key, @JsonProperty(value="quota", required=true) int quota, @JsonProperty(value="endDate", required=true) LocalDate endDate) {
            this.key = key;
            this.quota = quota;
            this.endDate = endDate;
        }
    }

    static class Snapshot {
        private final UUID latestChange;
        private final StatisticsInfo statistics;

        @JsonCreator
        public Snapshot(@JsonProperty(value="latestChange", required=true) UUID latestChange, @JsonProperty(value="statistics", required=true) StatisticsInfo statistics) {
            this.latestChange = latestChange;
            this.statistics = statistics;
        }

        public UUID getLatestChange() {
            return this.latestChange;
        }

        public StatisticsInfo getStatistics() {
            return this.statistics;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    static class StatisticsInfo {
        String licenseKey;
        Map<YearMonth, Set<String>> statistics;
        Map<LicenseEvent.LicenseEventType, LocalDate> licenseEvents;

        StatisticsInfo(@JsonProperty(value="licenseKey", required=true) String licenseKey, @JsonProperty(value="statistics", required=true) Map<YearMonth, List<String>> userIdsFromFile, @JsonProperty(value="licenseEvents", required=true) Map<LicenseEvent.LicenseEventType, LocalDate> licenseEvents) {
            this.licenseKey = licenseKey;
            this.statistics = this.copyMap(userIdsFromFile);
            this.licenseEvents = new HashMap<LicenseEvent.LicenseEventType, LocalDate>(licenseEvents);
        }

        Map<YearMonth, Set<String>> copyMap(Map<YearMonth, ? extends Collection<String>> map) {
            TreeMap<YearMonth, Set<String>> treeMap = new TreeMap<YearMonth, Set<String>>();
            for (Map.Entry<YearMonth, ? extends Collection<String>> month : map.entrySet()) {
                treeMap.put(month.getKey(), new LinkedHashSet<String>(month.getValue()));
            }
            return treeMap;
        }

        List<String> getUserEntries(YearMonth month) {
            Set entries = this.statistics.getOrDefault(month, Collections.emptySet());
            return new ArrayList<String>(entries);
        }

        void addUserEntry(YearMonth month, String payload) {
            this.statistics.computeIfAbsent(month, key -> new LinkedHashSet()).add(payload);
        }

        Map<String, LocalDate> getLatestLicenseEvents() {
            return this.licenseEvents.entrySet().stream().collect(Collectors.toMap(entry -> ((LicenseEvent.LicenseEventType)((Object)((Object)entry.getKey()))).name(), Map.Entry::getValue));
        }

        void setLicenseEvent(String eventName, LocalDate latestOccurrence) {
            this.licenseEvents.put(LicenseEvent.LicenseEventType.valueOf(eventName), latestOccurrence);
        }
    }

    static class LicenseInfoWrapper {
        final LicenseInfo content;
        final String checksum;

        @JsonCreator
        LicenseInfoWrapper(@JsonProperty(value="content", required=true) LicenseInfo content, @JsonProperty(value="checksum", required=true) String checksum) {
            this.content = content;
            this.checksum = checksum;
        }
    }

    static class StatisticsInfoWrapper {
        final StatisticsInfo content;
        final String checksum;

        @JsonCreator
        StatisticsInfoWrapper(@JsonProperty(value="content", required=true) StatisticsInfo content, @JsonProperty(value="checksum", required=true) String checksum) {
            this.content = content;
            this.checksum = checksum;
        }
    }
}

