package com.unity3d.ads.adplayer

import android.view.ViewGroup
import android.webkit.WebView
import com.unity3d.ads.core.domain.SendWebViewClientErrorDiagnostics
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow

class AndroidWebViewContainer(
    val webView: WebView,
    private val webViewClient: AndroidWebViewClient,
    private val sendWebViewClientErrorDiagnostics: SendWebViewClientErrorDiagnostics,
) : WebViewContainer {
    private val mainScope = MainScope() + CoroutineName("AndroidWebViewContainer")
    private val isDestroyed = MutableStateFlow(false)

    override suspend fun loadUrl(url: String) {
        withContext(mainScope.coroutineContext) {
            webView.loadUrl(url)
        }

        val loadResult = webViewClient.onLoadFinished.await()
        if (loadResult.isNotEmpty()) {
            destroy()
            sendWebViewClientErrorDiagnostics(loadResult)
            throw LoadWebViewError(loadResult)
        }
    }

    override suspend fun evaluateJavascript(script: String) {
        if (isDestroyed.value) return

        withContext(mainScope.coroutineContext) {
            webView.evaluateJavascript("javascript:$script", null)
        }
    }

    override suspend fun addJavascriptInterface(webViewBridgeInterface: WebViewBridge, name: String) {
        if (isDestroyed.value) return

        withContext(mainScope.coroutineContext) {
            webView.addJavascriptInterface(webViewBridgeInterface, name)
        }
    }

    override suspend fun destroy() {
        if (isDestroyed.value) return

        withContext(mainScope.coroutineContext) {
            (webView.parent as? ViewGroup)?.removeView(webView)
            webView.destroy()
            isDestroyed.value = true
        }

        mainScope.cancel()
    }
}