package com.unity3d.services

import android.content.Context
import com.unity3d.ads.IUnityAdsLoadListener
import com.unity3d.ads.IUnityAdsTokenListener
import com.unity3d.ads.TokenConfiguration
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.UnityAdsShowOptions
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.model.Listeners
import com.unity3d.ads.core.domain.GetAdObject
import com.unity3d.ads.core.domain.GetAsyncHeaderBiddingToken
import com.unity3d.ads.core.domain.GetGameId
import com.unity3d.ads.core.domain.GetHeaderBiddingToken
import com.unity3d.ads.core.domain.GetInitializationState
import com.unity3d.ads.core.domain.InitializeBoldSDK
import com.unity3d.ads.core.domain.LegacyLoadUseCase
import com.unity3d.ads.core.domain.LegacyShowUseCase
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.BANNER_DESTROYED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNCAUGHT_EXCEPTION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SOURCE_GET_TOKEN_API
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SOURCE_LOAD_API
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SOURCE_PUBLIC_API
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYNC
import com.unity3d.ads.core.domain.ShouldAllowInitialization
import com.unity3d.ads.core.domain.TokenNumberProvider
import com.unity3d.ads.core.domain.om.OmFinishSession
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.getShortenedStackTrace
import com.unity3d.ads.core.log.Logger
import com.unity3d.services.banners.UnityBannerSize
import com.unity3d.services.core.di.IServiceComponent
import com.unity3d.services.core.di.IServiceProvider
import com.unity3d.services.core.di.ServiceProvider
import com.unity3d.services.core.di.ServiceProvider.NAMED_GET_TOKEN_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_LOAD_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_OMID_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_SHOW_SCOPE
import com.unity3d.services.core.di.get
import com.unity3d.services.core.di.inject
import com.unity3d.services.core.domain.task.EmptyParams
import com.unity3d.services.core.domain.task.InitializeSDK
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.properties.SdkProperties
import com.unity3d.services.core.properties.Session.Default.id
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.updateAndGet
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import java.util.concurrent.locks.ReentrantLock
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource


/**
 * SDK Kotlin entry point
 */
@OptIn(ExperimentalTime::class)
class UnityAdsSDK(private val serviceProvider: IServiceProvider = ServiceProvider) : IServiceComponent {

    override fun getServiceProvider(): IServiceProvider {
        return serviceProvider
    }

    // Used in `initialize` which is thread safe because of @Synchronized
    private var initializationJob: Job? = null

    /**
     * Initialize the SDK
     */
    fun initialize(gameId: String?, source: String = SOURCE_PUBLIC_API): Job = synchronized(this) {
        // If initialization is already in progress, return the existing job
        val currentJob = initializationJob
        if (currentJob != null && currentJob.isActive) {
            return currentJob
        }

        val shouldAllowInitialization: ShouldAllowInitialization by inject()

        if (!shouldAllowInitialization(gameId)) return Job()

        val alternativeFlowReader: AlternativeFlowReader by inject()
        val initializeSDK: InitializeSDK by inject()
        val initializeBoldSDK: InitializeBoldSDK by inject()

        val initScope = get<CoroutineScope>(NAMED_INIT_SCOPE)

        // Initialization status will be updated from one of these init methods.
        val newInitializationJob = initScope.launch {
            val isAlternativeFlowEnabled = alternativeFlowReader()
            if (isAlternativeFlowEnabled) {
                initializeBoldSDK(source = source)
            } else {
                //TODO sessionId is wrong
                val mode = if (SdkProperties.isTestMode()) "test mode" else "production mode"
                DeviceLog.info("Initializing Unity Services ${SdkProperties.getVersionName()} (${SdkProperties.getVersionCode()}) with game id $gameId in $mode, session $id")

                initializeSDK(EmptyParams)
            }

            initScope.cancel()
        }

        initializationJob = newInitializationJob

        return newInitializationJob
    }

    fun load(
        placementId: String?,
        loadOptions: UnityAdsLoadOptions,
        listener: IUnityAdsLoadListener?,
        bannerSize: UnityBannerSize? = null
    ): Job {
        val getGameId: GetGameId by inject()
        initialize(getGameId(), source = SOURCE_LOAD_API)

        val loadScope = get<CoroutineScope>(NAMED_LOAD_SCOPE)
        val context: Context by inject()

        return loadScope.launch {
            val loadBoldSDK = get<LegacyLoadUseCase>()
            loadBoldSDK(context, placementId, loadOptions, listener, bannerSize)
            loadScope.cancel()
        }
    }

    fun show(placementId: String?, showOptions: UnityAdsShowOptions?, listener: Listeners): Job {
        val showScope = get<CoroutineScope>(NAMED_SHOW_SCOPE)
        val showBoldSDK = get<LegacyShowUseCase>()

        return showScope.launch {
            showBoldSDK(
                placement = placementId,
                unityAdsShowOptions = showOptions,
                listeners = listener
            )
            showScope.cancel()
        }
    }

    fun getToken(): String? = runBlocking {
        fetchToken(true.toString())
    }

    fun getToken(listener: IUnityAdsTokenListener?): Job {
        return getToken(null, listener)
    }

    fun getToken(tokenConfiguration: TokenConfiguration?, listener: IUnityAdsTokenListener?): Job {
        val getGameId: GetGameId by inject()
        initialize(getGameId(), source = SOURCE_GET_TOKEN_API)

        val tokenNumberProvider: TokenNumberProvider by inject()
        val getAsyncHeaderBiddingToken: GetAsyncHeaderBiddingToken by inject()
        val getTokenScope = get<CoroutineScope>(NAMED_GET_TOKEN_SCOPE)

        return getTokenScope.launch {
            getAsyncHeaderBiddingToken(tokenNumberProvider(), tokenConfiguration, listener)
            getTokenScope.cancel()
        }
    }

    private suspend fun fetchToken(sync: String): String? {
        val tokenNumberProvider: TokenNumberProvider by inject()
        val getHeaderBiddingToken: GetHeaderBiddingToken by inject()
        val getInitializationState: GetInitializationState by inject()
        val sendDiagnosticEvent: SendDiagnosticEvent by inject()
        val logger: Logger by inject()
        val startTime = TimeSource.Monotonic.markNow()
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_STARTED,
            tags = mapOf(
                SYNC to sync,
                STATE to getInitializationState().toString()
            ))
        var reason: String? = null
        var reasonDebug: String? = null
        val token = if (getInitializationState() != InitializationState.INITIALIZED){
            reason = REASON_NOT_INITIALIZED
            null
        } else {
            try {
                runBlocking { getHeaderBiddingToken(tokenNumberProvider()) }
            } catch (e: Exception) {
                reason = REASON_UNCAUGHT_EXCEPTION
                reasonDebug = e.getShortenedStackTrace()
                null
            }
        }

        sendDiagnosticEvent(
            event = if (token == null) SendDiagnosticEvent.HB_FAILURE else SendDiagnosticEvent.HB_SUCCESS,
            value = startTime.elapsedMillis(),
            tags = buildMap {
                put(SYNC, sync)
                put(STATE, getInitializationState().toString())
                reason?.let{ put(REASON, reason) }
                reasonDebug?.let{ put(REASON_DEBUG, reasonDebug) }
            },
            tokenNumber = tokenNumberProvider()
        )

        if(token == null) {
            logger.error("Returned nil token due to: $reason")
        } else {
            logger.info("Generated a valid token.")
        }

        return token
    }

    fun finishOMIDSession(opportunityId: String): Job {
        val alternativeFlowReader: AlternativeFlowReader by inject()
        if (!alternativeFlowReader()) return Job().apply { complete() }

        val getAdObject: GetAdObject by inject()
        val omFinishSession: OmFinishSession by inject()
        val omidScope = get<CoroutineScope>(NAMED_OMID_SCOPE)
        return omidScope.launch {
            val adObject = getAdObject(opportunityId)
            adObject?.let { omFinishSession(it) }
            omidScope.cancel()
        }
    }

    fun sendBannerDestroyed() {
        val alternativeFlowReader: AlternativeFlowReader by inject()
        if (!alternativeFlowReader()) return

        val sendDiagnosticEvent: SendDiagnosticEvent by inject()
        sendDiagnosticEvent(BANNER_DESTROYED)
    }
}