package com.unity3d.ads.core.log

import android.util.Log
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.CreateFile

internal class UnityLogger(
    sessionRepository: SessionRepository,
    createFile: CreateFile,
) : Logger {

    companion object {
        private const val LOG_TAG = "UnityAds"
    }

    override var logLevel = LogLevel.INFO
        set(value) { if (!isForced) field = value }

    /**
     * Set to true if the logger was forced to log in trace mode. This is used to prevent
     * the log level from being changed after the logger was initialized with TRACE.
     */
    private val isForced: Boolean

    init {
        val forceDebugFile = createFile("/data/local/tmp/UnityAdsForceDebugMode")

        try {
            val enableTracing = sessionRepository.nativeConfiguration.debugSettings.enableTracing
            if (forceDebugFile.exists() || enableTracing) {
                logLevel = LogLevel.TRACE
                debug("Unity logger initialized in TRACE mode")
            }
        } catch (_: Throwable) {
            Log.w(LOG_TAG, "An issue occurred while initializing the logger")
        } finally {
            isForced = logLevel == LogLevel.TRACE
        }
    }

    private fun canLog(targetLogLevel: Int) = targetLogLevel >= logLevel.toAndroidLogLevel()


    override fun info(message: String) {
        if (!canLog(Log.INFO)) return
        Log.i(LOG_TAG, message)
    }

    override fun debug(message: String) {
        if (!canLog(Log.DEBUG)) return
        Log.d(LOG_TAG, message)
    }

    override fun error(message: String, e: Throwable?) {
        if (!canLog(Log.ERROR)) return
        Log.e(LOG_TAG, message, e)
    }

    override fun trace(message: String, e: Throwable?) {
        if (!canLog(Log.VERBOSE)) return
        Log.v(LOG_TAG, message, e)
    }

    override fun debug(fb: () -> String) {
        if (!canLog(Log.DEBUG)) return
        debug(fb())
    }
}