package com.unity3d.ads.core.domain.events

import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.ads.core.data.repository.TransactionEventRepository
import com.unity3d.ads.core.domain.GetRequestPolicy
import com.unity3d.ads.core.domain.GetUniversalRequestForPayLoad
import com.unity3d.ads.gatewayclient.GatewayClient
import gatewayprotocol.v1.UniversalRequestKt
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach

class TransactionEventObserver(
    private val getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
    private val scope: CoroutineScope,
    private val transactionEventRepository: TransactionEventRepository,
    private val gatewayClient: GatewayClient,
    private val getRequestPolicy: GetRequestPolicy,
    private val iapTransactionStore: ByteStringDataSource,
) {
    private val isRunning = MutableStateFlow(false)

    operator fun invoke() {
        // If we are already collecting events, do nothing
        if (isRunning.getAndUpdate { true }) return
        transactionEventRepository.transactionEvents.onEach { currentTransactionEventRequest ->
            try {
                val payload = UniversalRequestKt.payload {
                    transactionEventRequest = currentTransactionEventRequest
                }
                val fullRequest = getUniversalRequestForPayLoad(payload)
                val response = gatewayClient.request(
                    request = fullRequest,
                    requestPolicy = getRequestPolicy(),
                    operationType = OperationType.TRANSACTION_EVENT
                )
                if (response.error != null) {
                    // Set transaction timestamp value in the iapTransactionStore
                    // We can use the one stored in the request since it matches when it was created.
                    iapTransactionStore.set(fullRequest.sharedData.timestamps.timestamp.toByteString())
                }
            } catch (exception: UnityAdsNetworkException) {
                // Don't do anything for network exceptions like timeout or gateway errors since its best effort to collect.
                // We still catch them to avoid stopping the sending of future events.
            }
        }.launchIn(scope)
    }
}