package com.unity3d.ads.core.data.manager

import com.google.protobuf.Timestamp
import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.model.exception.TransactionException
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.data.repository.TransactionEventRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.billing.IsBillingClientAvailable
import com.unity3d.ads.core.domain.billing.ProductDetailsFetcher
import com.unity3d.ads.core.domain.billing.ProductDetailsResult
import com.unity3d.ads.core.domain.events.GetTransactionData
import com.unity3d.ads.core.domain.events.GetTransactionRequest
import com.unity3d.ads.core.log.Logger
import com.unity3d.services.store.gpbl.BillingResultResponseCode
import com.unity3d.services.store.gpbl.bridges.BillingResultBridge
import com.unity3d.services.store.gpbl.bridges.PurchaseBridge
import com.unity3d.services.store.gpbl.bridges.billingclient.BillingClientAdapter
import com.unity3d.services.store.gpbl.listeners.BillingInitializationListener
import gatewayprotocol.v1.TransactionEventRequestOuterClass.TransactionData
import gatewayprotocol.v1.TransactionEventRequestOuterClass.TransactionOrigin
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.time.Duration.Companion.milliseconds

// Deals with querying purchases from the Google Play Billing Library.
class TransactionEventManager(
    private val scope: CoroutineScope,
    private val billingClientAdapter: BillingClientAdapter?,
    private val getTransactionData: GetTransactionData,
    private val getTransactionRequest: GetTransactionRequest,
    private val transactionEventRepository: TransactionEventRepository,
    private val iapTransactionStore: ByteStringDataSource,
    private val isBillingClientAvailable: IsBillingClientAvailable,
    private val sessionRepository: SessionRepository,
    private val productDetailsFetcher: ProductDetailsFetcher,
    private val logger: Logger
) {
    operator fun invoke() {
        if (!isBillingClientAvailable() || billingClientAdapter == null) {
            logger.trace(BILLING_SERVICE_UNAVAILABLE)
            return
        }
        scope.launch {
            if (!billingClientAdapter.isInitialized) {
                try {
                    suspendCancellableCoroutine {
                        billingClientAdapter.initialize(object : BillingInitializationListener {
                            val hasBeenResumed = MutableStateFlow(false)
                            private fun tryResume() {
                                if (it.isActive && !hasBeenResumed.getAndUpdate { true }) it.resume(Unit)
                            }

                            private fun tryResumeWithException(exception: Exception) {
                                if (it.isActive && !hasBeenResumed.getAndUpdate { true }) it.resumeWithException(exception)
                            }

                            override fun onIsAlreadyInitialized() {
                                tryResume()
                            }

                            override fun onBillingSetupFinished(billingResult: BillingResultBridge) {
                                if (billingResult.responseCode != BillingResultResponseCode.OK) {
                                    tryResumeWithException(TransactionException("Billing setup failed"))
                                    return
                                }
                                tryResume()
                            }

                            override fun onBillingServiceDisconnected() {
                                tryResumeWithException(TransactionException("Billing service disconnected"))
                            }

                            override fun onPurchaseUpdated(
                                billingResult: BillingResultBridge,
                                purchases: List<PurchaseBridge>?
                            ) {
                                onPurchasesReceived(billingResult, purchases, TransactionOrigin.TRANSACTION_ORIGIN_LIVE_UPDATE)
                            }
                        })
                    }
                    // Query historical purchases only if the feature flag is enabled
                    // Note that this only sends active subscriptions and non-consumed in-app purchases
                    // Previous purchases that are not active anymore or have been consumed will not be sent
                    if (sessionRepository.nativeConfiguration.featureFlags.shouldSendIapHistory) {
                        billingClientAdapter.queryPurchasesAsync(INAPP) { billingResult, purchases ->
                            onPurchasesReceived(
                                billingResult,
                                purchases,
                                TransactionOrigin.TRANSACTION_ORIGIN_HISTORICAL
                            )
                        }
                        billingClientAdapter.queryPurchasesAsync(SUBS) { billingResult, purchases ->
                            onPurchasesReceived(
                                billingResult,
                                purchases,
                                TransactionOrigin.TRANSACTION_ORIGIN_HISTORICAL
                            )
                        }
                    }

                } catch (e: Exception) {
                    logger.trace(BILLING_SERVICE_UNAVAILABLE, e)
                }
            }
        }

    }

    private fun onPurchasesReceived(billingResult: BillingResultBridge, purchases: List<PurchaseBridge>?, transactionOrigin: TransactionOrigin) {
        if (billingResult.responseCode != BillingResultResponseCode.OK || purchases.isNullOrEmpty() || billingClientAdapter == null) return
        scope.launch {
            val transactionDataList = mutableListOf<TransactionData>()
            val deferredPurchaseList = purchases.map { CompletableDeferred<Unit>() }
            purchases.forEachIndexed { index, purchase ->
                try {
                    val purchaseTime = purchase.originalJson.optLong("purchaseTime", -1).takeIf { it >= 0 }?.milliseconds
                    val productId = purchase.originalJson.optString("productId").takeIf { it.isNotBlank() }
                    
                    if (purchaseTime == null || productId.isNullOrBlank()) {
                        deferredPurchaseList[index].complete(Unit)
                        return@forEachIndexed
                    }
                    
                    val lastIapTransactionSent = Timestamp.parseFrom(iapTransactionStore.get().data)

                    if (lastIapTransactionSent.seconds < purchaseTime.inWholeSeconds) {
                        val productDetailsResult = productDetailsFetcher.fetchProductDetails(productId)
                        
                        if (productDetailsResult is ProductDetailsResult.Success) {
                            transactionDataList.add(getTransactionData(purchase, productDetailsResult.productDetailsJson))
                        }
                    }
                    deferredPurchaseList[index].complete(Unit)
                } catch (e: Exception) {
                    deferredPurchaseList[index].complete(Unit)
                }
            }

            // Wait for all purchases to complete
            awaitAll(*deferredPurchaseList.toTypedArray())

            if (transactionDataList.isNotEmpty()) {
                val transactionRequest = getTransactionRequest(transactionDataList, billingClientAdapter.getAdapterVersion(), transactionOrigin)
                transactionEventRepository.addTransactionEvent(transactionRequest)
            }
        }
    }

    companion object {
        private const val INAPP = "inapp"
        private const val SUBS = "subs"
        const val BILLING_SERVICE_UNAVAILABLE = "Billing client is not available"
    }
}