package com.unity3d.ads.core.domain

import android.content.Context
import com.google.protobuf.ByteString
import com.google.protobuf.kotlin.isNotEmpty
import com.unity3d.ads.UnityAds.UnityAdsLoadError
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_AD_OBJECT
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_COMMUNICATION_FAILURE
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_TIMEOUT
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.exception.GatewayException
import com.unity3d.ads.core.data.model.exception.NetworkTimeoutException
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_CONFIG_FAILURE_TIME
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_CONFIG_SUCCESS_TIME
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_OBJECT_NOT_FOUND
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.services.UnityAdsConstants.Messages.MSG_INTERNAL_ERROR
import gatewayprotocol.v1.AdRequestOuterClass.BannerSize
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType.DIAGNOSTIC_AD_TYPE_BANNER
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType.DIAGNOSTIC_AD_TYPE_FULLSCREEN
import gatewayprotocol.v1.HeaderBiddingAdMarkupOuterClass.HeaderBiddingAdMarkup
import gatewayprotocol.v1.AdFormatOuterClass.AdFormat
import gatewayprotocol.v1.adResponse
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.measureTimedValue

internal class AndroidLoad(
    private val defaultDispatcher: CoroutineDispatcher,
    private val getAdRequest: GetAdRequest,
    private val getAdPlayerConfigRequest: GetAdPlayerConfigRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val handleGatewayAdResponse: HandleGatewayAdResponse,
    private val sessionRepository: SessionRepository,
    private val gatewayClient: GatewayClient,
    private val adRepository: AdRepository,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
) : Load {
    @OptIn(ExperimentalTime::class)
    override suspend fun invoke(
        context: Context,
        placement: String,
        opportunityId: ByteString,
        headerBiddingAdMarkup: HeaderBiddingAdMarkup,
        bannerSize: BannerSize?,
        loadOptions: UnityAdsLoadOptions
    ): LoadResult = withContext(defaultDispatcher) {
        try {
            if (!sessionRepository.isSdkInitialized) {
                return@withContext LoadResult.Failure(error = UnityAdsLoadError.INITIALIZE_FAILED, reason = REASON_NOT_INITIALIZED)
            }

            val isBanner = bannerSize != null
            val adType = if (isBanner) DIAGNOSTIC_AD_TYPE_BANNER else DIAGNOSTIC_AD_TYPE_FULLSCREEN
            val isHeaderBidding = !headerBiddingAdMarkup.adData.isEmpty
            val tmpAdObject = getTmpAdObject(opportunityId, placement, isHeaderBidding, adType, loadOptions)
            val response = if (!isHeaderBidding) {
                // waterfall
                incrementLoadRequestCount(isBanner)
                val loadAdRequest = getAdRequest(placement, opportunityId, bannerSize)
                val requestPolicy = getRequestPolicy()
                val gatewayAdResponse = measureTimedValue {
                    runCatching {
                        gatewayClient.request(
                            request = loadAdRequest,
                            requestPolicy = requestPolicy,
                            operationType = OperationType.LOAD
                        )
                    }
                }.also { (result, duration) ->
                    sendDiagnosticEvent(
                        if (result.isSuccess) LOAD_CONFIG_SUCCESS_TIME else LOAD_CONFIG_FAILURE_TIME,
                        duration.toDouble(DurationUnit.MILLISECONDS),
                        adObject = tmpAdObject,
                    )
                }.value.getOrThrow()
                gatewayAdResponse.payload.adResponse
            } else {
                // header bidding
                incrementLoadRequestAdmCount(isBanner)
                val adPlayerConfigRequest = getAdPlayerConfigRequest(placement, opportunityId, headerBiddingAdMarkup.configurationToken, bannerSize?.let { AdFormat.AD_FORMAT_BANNER })
                val requestPolicy = getRequestPolicy()
                val gatewayAdPlayerConfigResponse = measureTimedValue {
                    runCatching {
                        gatewayClient.request(
                            request = adPlayerConfigRequest,
                            requestPolicy = requestPolicy,
                            operationType = OperationType.LOAD_HEADER_BIDDING
                        )
                    }
                }.also { (result, duration) ->
                    sendDiagnosticEvent(
                        if (result.isSuccess) LOAD_CONFIG_SUCCESS_TIME else LOAD_CONFIG_FAILURE_TIME,
                        duration.toDouble(DurationUnit.MILLISECONDS),
                        adObject = tmpAdObject,
                    )
                }.value.getOrThrow()
                if (gatewayAdPlayerConfigResponse.hasError()) {
                    return@withContext LoadResult.Failure(
                        error = UnityAdsLoadError.INTERNAL_ERROR,
                        message = MSG_INTERNAL_ERROR,
                        reason = REASON_GATEWAY,
                        reasonDebug = gatewayAdPlayerConfigResponse.error.errorText)
                }
                val response = gatewayAdPlayerConfigResponse.payload.adPlayerConfigResponse
                adResponse {
                    adData = headerBiddingAdMarkup.adData
                    adDataVersion = headerBiddingAdMarkup.adDataVersion
                    trackingToken = response.trackingToken
                    impressionConfiguration = response.impressionConfiguration
                    impressionConfigurationVersion = response.impressionConfigurationVersion
                    webviewConfiguration = response.webviewConfiguration
                    adDataRefreshToken = response.adDataRefreshToken
                    if (response.hasError()) {
                        error = response.error
                    }
                    if (response.adData.isNotEmpty()) {
                        adData = response.adData
                        adDataVersion = response.adDataVersion
                    }
                }
            }

            val adLoadResponse = handleGatewayAdResponse(loadOptions, opportunityId, response, context, placement, adType, isHeaderBidding)
            when (adLoadResponse) {
                is LoadResult.Success -> {
                    val adObject = adRepository.getAd(opportunityId)
                    if (adObject == null) {
                        LoadResult.Failure(error = UnityAdsLoadError.INTERNAL_ERROR, message = MSG_AD_OBJECT, reason = REASON_AD_OBJECT_NOT_FOUND)
                    } else {
                        LoadResult.Success(adObject = adObject)
                    }
                }
                is LoadResult.Failure -> adLoadResponse
            }
        } catch (e: UnityAdsNetworkException) {
            handleGatewayException(e)
        }
    }

    private fun handleGatewayException(e: UnityAdsNetworkException): LoadResult.Failure {
        return LoadResult.Failure(
            error = when (e) {
                is NetworkTimeoutException -> UnityAdsLoadError.TIMEOUT
                else -> UnityAdsLoadError.INTERNAL_ERROR
            },
            message = when (e) {
                is NetworkTimeoutException -> MSG_TIMEOUT
                is GatewayException -> e.message
                else -> MSG_COMMUNICATION_FAILURE
            },
            reason = when (e) {
                is NetworkTimeoutException -> REASON_TIMEOUT
                else -> REASON_GATEWAY
            },
            reasonDebug = e.message,
            throwable = e
        )
    }

    private fun incrementLoadRequestCount(isBanner: Boolean) {
        if (isBanner) {
            sessionRepository.incrementBannerLoadRequestCount()
        } else {
            sessionRepository.incrementLoadRequestCount()
        }
    }

    private fun incrementLoadRequestAdmCount(isBanner: Boolean) {
        if (isBanner) {
            sessionRepository.incrementBannerLoadRequestAdmCount()
        } else {
            sessionRepository.incrementLoadRequestAdmCount()
        }
    }

    private fun getTmpAdObject(
        opportunityId: ByteString,
        placement: String,
        isHeaderBidding: Boolean,
        adType: DiagnosticAdType,
        loadOptions: UnityAdsLoadOptions
    ): AdObject {
        return AdObject(
            opportunityId = opportunityId,
            placementId = placement,
            trackingToken = ByteString.EMPTY,
            adPlayer = null,
            loadOptions = loadOptions,
            isHeaderBidding = isHeaderBidding,
            adType = adType
        )
    }
}