package com.unity3d.ads.core.data.repository

import android.app.Activity
import com.unity3d.ads.core.domain.AndroidGetLifecycleFlow
import com.unity3d.ads.core.domain.LifecycleEvent
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.shareIn
import java.lang.Exception
import java.lang.ref.WeakReference

sealed interface FocusState {
    val activity: WeakReference<Activity>

    data class Focused(override val activity: WeakReference<Activity>) : FocusState
    data class Unfocused(override val activity: WeakReference<Activity>) : FocusState
}

class FocusRepository(
    getLifecycleFlow: AndroidGetLifecycleFlow,
    defaultDispatcher: CoroutineDispatcher,
) {
    val focusState: SharedFlow<FocusState> = try {
        getLifecycleFlow()
            .filter { it is LifecycleEvent.Paused || it is LifecycleEvent.Resumed }
            .map {
                if (it is LifecycleEvent.Resumed) {
                    FocusState.Focused(it.activity)
                } else {
                    FocusState.Unfocused(it.activity)
                }
            }
            .shareIn(CoroutineScope(defaultDispatcher), SharingStarted.Eagerly, 1)
    } catch (e: Exception) {
        DeviceLog.error("Could not subscribe to lifecycle flow", e)
        MutableSharedFlow()
    }
}