package com.unity3d.ads.adplayer

import android.content.Context
import android.content.Intent
import com.google.protobuf.kotlin.toByteStringUtf8
import com.unity3d.ads.adplayer.AdPlayer.Companion.OFFERWALL_EVENT_QUEUE_SIZE
import com.unity3d.ads.adplayer.AdPlayer.Companion.SCAR_EVENT_QUEUE_SIZE
import com.unity3d.ads.core.data.datasource.VolumeSettingsChange
import com.unity3d.ads.core.data.manager.OfferwallManager
import com.unity3d.ads.core.data.manager.ScarManager
import com.unity3d.ads.core.data.model.ScarEvent
import com.unity3d.ads.core.data.model.OfferwallShowEvent
import com.unity3d.ads.core.data.model.SessionChange
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.OpenMeasurementRepository
import com.unity3d.ads.core.data.repository.OrientationRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_SESSION_FINISH_DELAY_MS
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.flow.shareIn
import kotlinx.coroutines.launch
import org.json.JSONObject
import java.util.UUID

/**
 * An internal class representing an Android fullscreen webview ad player.
 *
 * This class implements the [AdPlayer] via [WebViewAdPlayer] and [FullscreenAdPlayer] interfaces. It provides methods to show
 * the ad, destroy the ad player, and send various events related to the ad player.
 *
 * @property webViewAdPlayer The underlying webview ad player.
 * @property opportunityId The ID of the ad opportunity.
 * @property webViewContainer The Android webview container.
 * @property deviceInfoRepository The repository for device information.
 * @property sessionRepository The repository for session information.
 * @property orientationRepository the repository for resumed activity set orientation
 */
internal class AndroidFullscreenWebViewAdPlayer(
    private val webViewAdPlayer: WebViewAdPlayer,
    private val opportunityId: String,
    override val webViewContainer: AndroidWebViewContainer,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val sessionRepository: SessionRepository,
    private val openMeasurementRepository: OpenMeasurementRepository,
    private val scarManager: ScarManager,
    private val offerwallManager: OfferwallManager,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val adRepository: AdRepository,
    private val orientationRepository: OrientationRepository,
    private val context: Context,
) : AdPlayer by webViewAdPlayer, FullscreenAdPlayer {
    private val adObject by lazy {
        runCatching { adRepository.getAd(UUID.fromString(opportunityId).toByteString()) }.getOrNull()
    }

    /**
     * Show method is responsible for showing the [FullScreenWebViewDisplay], which is an activity. It will contain the webview (using [AndroidWebViewContainer])
     * and the ad will be played in the webview via the ad viewer.
     *
     * @param showOptions The show options for displaying the activity. They are not the one passed to the ad viewer.
     * @throws IllegalArgumentException if the showOptions is not an instance of AndroidShowOptions. (KMM future compatibility)
     */
    override fun show(showOptions: ShowOptions) {
        require(showOptions is AndroidShowOptions)
        val isScarAd = showOptions.isScarAd
        val isOfferwallAd = showOptions.isOfferwallAd

        // Captures messages from the display (activity on Android) and routes them to the webview ad player.
        val listenerStarted = CompletableDeferred<Unit>()
        displayMessages
            .onSubscription { listenerStarted.complete(Unit) }
            .filter { it.opportunityId == opportunityId }
            .onEach(::displayEventsRouter)
            .launchIn(scope)

        deviceInfoRepository.volumeSettingsChange
            .onEach(::handleVolumeSettingsChange)
            .launchIn(scope)

        webViewAdPlayer.onShowEvent
            .filter { it is ShowEvent.Completed || it is ShowEvent.Error }
            .onEach { destroy() }
            .launchIn(scope)

        sessionRepository.onChange
            .onEach(::handleSessionChange)
            .launchIn(scope)

        if (!isScarAd && !isOfferwallAd) {
            sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN, adObject = adObject)

            val intent = Intent(context, FullScreenWebViewDisplay::class.java).also {
                it.putExtra("opportunityId", opportunityId)
                showOptions.unityAdsShowOptions?.let { showOptionsData ->
                    it.putExtra("showOptions", JSONObject(showOptionsData).toString())
                }
                it.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION or Intent.FLAG_ACTIVITY_NEW_TASK)
                it.putExtra("orientation", orientationRepository.resumedActivityOrientation.value)
            }

            scope.launch {
               listenerStarted.await() 
                context.startActivity(intent)
                sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT, adObject = adObject)
            }
        } else if (isScarAd){

            /**
             * SCAR / AdPlayer "event dance" flow:
             * 1. Request show from SCAR and start collecting events
             * 2. Let AdPlayer know that SCAR is ready to show
             * 3. Collect SCAR events and map them to GMA events when AdPlayer is ready to receive them
             */
            val scarEvents = scarManager.show(showOptions.placementId ?: "", showOptions.scarQueryId ?: "")
                .shareIn(scope, SharingStarted.Eagerly, SCAR_EVENT_QUEUE_SIZE)

            scope.launch {
                onScarEvent
                    .onStart { displayMessages.emit(DisplayMessage.DisplayReady(opportunityId, showOptions.unityAdsShowOptions)) }
                    .first { it == ScarEvent.Show }

                scarEvents
                    .mapNotNull { it.gmaEvent }
                    .collect(webViewAdPlayer::sendGmaEvent)

            }

        } else {
            // Offerwall ad
            scope.launch {
                onOfferwallEvent
                    .onStart { displayMessages.emit(DisplayMessage.DisplayReady(opportunityId, showOptions.unityAdsShowOptions)) }
                    .first { it == OfferwallShowEvent.Show }

                offerwallManager.showAd(showOptions.offerwallPlacementName ?: "")
                    .shareIn(scope, SharingStarted.Eagerly, OFFERWALL_EVENT_QUEUE_SIZE)
                    .mapNotNull { it.offerwallEvent }
                    .collect(webViewAdPlayer::sendOfferwallEvent)
            }
        }
    }

    private suspend fun handleVolumeSettingsChange(change: VolumeSettingsChange) {
        when (change) {
            is VolumeSettingsChange.MuteChange -> webViewAdPlayer.sendMuteChange(change.isMuted)
            is VolumeSettingsChange.VolumeChange -> webViewAdPlayer.sendVolumeChange(change.volume)
        }
    }

    private suspend fun handleSessionChange(change: SessionChange) {
        when (change) {
            is SessionChange.UserConsentChange -> webViewAdPlayer.sendUserConsentChange(change.value.toByteArray())
            is SessionChange.PrivacyFsmChange -> webViewAdPlayer.sendPrivacyFsmChange(change.value.toByteArray())
        }
    }

    override suspend fun destroy() {
        // Notify the display (activity on Android) to finish (destroy itself).
        displayMessages.emit(DisplayMessage.DisplayFinishRequest(opportunityId))

        // If the if had an OMID session we need to delay webview cleanup for 1 second.
        // https://interactiveadvertisingbureau.github.io/Open-Measurement-SDKAndroid/#9-stop-the-session
        if (openMeasurementRepository.hasSessionFinished(opportunityId.toByteStringUtf8())) {
            delay(OM_SESSION_FINISH_DELAY_MS)
        }

        // Destroy the webview container.
        webViewContainer.destroy()

        super<AdPlayer>.destroy()
    }

    private fun displayEventsRouter(displayMessage: DisplayMessage) = scope.launch {
        sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_ACTIVITY_EVENT, tags = mapOf(
            "eventType" to displayMessage::class.simpleName.toString(),
            "content" to when (displayMessage) {
                is DisplayMessage.VisibilityChanged -> displayMessage.isVisible.toString()
                is DisplayMessage.FocusChanged -> displayMessage.isFocused.toString()
                else -> ""
            }
        ), adObject = adObject)

        when (displayMessage) {
            is DisplayMessage.DisplayReady -> webViewAdPlayer.requestShow(displayMessage.showOptions)
            is DisplayMessage.WebViewInstanceRequest -> displayMessages.emit(DisplayMessage.WebViewInstanceResponse(displayMessage.opportunityId, webViewContainer.webView))
            is DisplayMessage.VisibilityChanged -> webViewAdPlayer.sendVisibilityChange(displayMessage.isVisible)
            is DisplayMessage.FocusChanged -> webViewAdPlayer.sendFocusChange(displayMessage.isFocused)
            is DisplayMessage.DisplayDestroyed -> webViewAdPlayer.sendActivityDestroyed()
            // todo: report error message?
            is DisplayMessage.DisplayError -> destroy()
            else -> Unit
        }
    }

    companion object {
        /**
         * A flow of display messages that is used to communicate with displays. It is currently used by Android with [FullScreenWebViewDisplay].
         */
        val displayMessages = MutableSharedFlow<DisplayMessage>()
    }
}
