package com.unity3d.ads.adplayer

import android.os.Build
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import androidx.annotation.RequiresApi
import androidx.webkit.WebResourceErrorCompat
import androidx.webkit.WebViewClientCompat
import androidx.webkit.WebViewFeature
import androidx.webkit.WebViewFeature.isFeatureSupported
import com.unity3d.ads.adplayer.model.WebViewClientError
import com.unity3d.ads.adplayer.model.ErrorReason
import com.unity3d.ads.core.domain.GetCachedAsset
import com.unity3d.ads.core.extensions.webResourceToErrorReason
import com.unity3d.ads.core.extensions.removeViewFromParent
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update

class AndroidWebViewClient(
    private val getCachedAsset: GetCachedAsset,
) : WebViewClientCompat() {
    private val loadErrors = MutableStateFlow(listOf<WebViewClientError>())
    private val _onLoadFinished = CompletableDeferred<List<WebViewClientError>>()
    val onLoadFinished: Deferred<List<WebViewClientError>> = _onLoadFinished

    private val _isRenderProcessGone = MutableStateFlow(false)
    val isRenderProcessGone = _isRenderProcessGone.asStateFlow()

    private val webviewType = MutableStateFlow<String>("")

    override fun onPageFinished(view: WebView, url: String) {
        if (url == BLANK_PAGE) {
            loadErrors.update { it + WebViewClientError(url, ErrorReason.REASON_WEB_BLANK) }
        }

        super.onPageFinished(view, url)
        _onLoadFinished.complete(loadErrors.value)
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    override fun onReceivedError(
        view: WebView,
        request: WebResourceRequest,
        error: WebResourceErrorCompat
    ) {
        super.onReceivedError(view, request, error)

        val reason = if (isFeatureSupported(WebViewFeature.WEB_RESOURCE_ERROR_GET_CODE)) {
            error.errorCode.webResourceToErrorReason()
        } else {
            ErrorReason.REASON_UNKNOWN
        }

        loadErrors.update {it + WebViewClientError(request.url.toString(), reason) }
    }

    override fun onReceivedHttpError(
        view: WebView,
        request: WebResourceRequest,
        errorResponse: WebResourceResponse
    ) {
        super.onReceivedHttpError(view, request, errorResponse)

        val error = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            WebViewClientError(
                url = request.url.toString(),
                reason = ErrorReason.REASON_WEB_ERROR_RECEIVED_HTTP,
                statusCode = errorResponse.statusCode
            )
        } else {
            WebViewClientError(reason = ErrorReason.REASON_WEB_ERROR_RECEIVED_HTTP)
        }

        loadErrors.update {it + error }
    }

    override fun shouldInterceptRequest(
        view: WebView,
        request: WebResourceRequest
    ): WebResourceResponse? {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            val url = request.url ?: return super.shouldInterceptRequest(view, request)

            val queryWebviewType = url.getQueryParameter("webviewType")
            if (!queryWebviewType.isNullOrBlank()) {
                webviewType.value = queryWebviewType
            }

            if (url.lastPathSegment == "favicon.ico") {
                return WebResourceResponse("image/png", null, null)
            }

            return getCachedAsset(request.url, webviewType.value)
        }
        return super.shouldInterceptRequest(view, request)
    }

    override fun onRenderProcessGone(view: WebView, detail: RenderProcessGoneDetail): Boolean {
        view.removeViewFromParent()
        view.destroy()
        // todo: if during load and show -> rebuild webview to not compromise show
        // todo: add diagnostic for render process gone during load/show

        // If onRenderProcessGone is called before the page is finished loading, complete the deferred with the error,
        // so it is reported as a load error.
        if (!_onLoadFinished.isCompleted) {
            loadErrors.update {
                it + WebViewClientError(
                    view.url.toString(),
                    ErrorReason.REASON_WEBVIEW_RENDER_PROCESS_GONE
                )
            }

            _onLoadFinished.complete(loadErrors.value)
        } else {
            _isRenderProcessGone.value = true
        }

        return true // returning false will kill the app
    }

    companion object {
        const val BLANK_PAGE = "about:blank"
    }
}