package com.unity3d.ads.core.domain

import android.app.Activity
import com.unity3d.ads.UnityAdsShowOptions
import com.unity3d.ads.adplayer.AndroidShowOptions
import com.unity3d.ads.core.configuration.GameServerIdReader
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.extensions.toBuiltInMap
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.onCompletion
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.flow.transformWhile
import java.lang.ref.WeakReference

class AndroidShow(
    private val adRepository: AdRepository,
    private val gameServerIdReader: GameServerIdReader,
    private val sendDiagnosticEvent: SendDiagnosticEvent
) : Show {
    override fun invoke(activity: Activity, adObject: AdObject, showOptions: UnityAdsShowOptions): Flow<ShowEvent> = flow {
        require(!adObject.opportunityId.isEmpty) { "No opportunityId" }

        val opportunityId = adObject.opportunityId

        val ad = adRepository.getAd(opportunityId) ?: throw IllegalStateException("No ad associated with opportunityId")

        sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_STARTED_AD_VIEWER, adObject = adObject)

        val adPlayer = ad.adPlayer ?: throw IllegalStateException("No adPlayer associated with ad")

        adPlayer.onShowEvent
            .onStart {
                ad.playerServerId = gameServerIdReader.readAndDelete(null)
                sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_EVENT_FLOW_STARTED, adObject = adObject)
                ad.adPlayer.show(
                    AndroidShowOptions(
                        WeakReference(activity),
                        showOptions.data?.toBuiltInMap(),
                        isScarAd = ad.isScarAd,
                        scarAdString = ad.scarAdString,
                        scarQueryId = ad.scarQueryId,
                        scarAdUnitId = ad.scarAdUnitId,
                        isOfferwallAd = ad.isOfferwallAd,
                        offerwallPlacementName = ad.offerwallPlacementName,
                        placementId = ad.placementId)
                    )
            }
            .onCompletion {
                sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_EVENT_FLOW_COMPLETED, adObject = adObject)
                adRepository.removeAd(opportunityId)
            }
            .transformWhile {
                emit(it)
                it !is ShowEvent.Completed && it !is ShowEvent.Error
            }
            .collect {
                sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_EVENT_FLOW_COLLECTED, adObject = adObject, tags = mapOf(
                    "event" to it::class.java.simpleName,
                ))
                emit(it)
            }
    }

    override suspend fun terminate(adObject: AdObject) {
        adObject.adPlayer?.destroy()
    }
}