package com.unity3d.ads.core.domain

import com.unity3d.ads.IUnityAdsTokenListener
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYNC
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.NOT_INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.FAILED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZING
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.AWAITED_INIT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_LISTENER_NULL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.retrieveUnityCrashValue
import com.unity3d.services.UnityAdsConstants.Timeout.GET_TOKEN_TIMEOUT_MS
import com.unity3d.services.core.misc.Utilities.wrapCustomerListener
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
class CommonInitAwaitingGetHeaderBiddingToken(
    val getHeaderBiddingToken: GetHeaderBiddingToken,
    val sendDiagnosticEvent: SendDiagnosticEvent,
    val getInitializationState: GetInitializationState,
    val awaitInitialization: AwaitInitialization,
) : GetAsyncHeaderBiddingToken {
    val startTime = TimeSource.Monotonic.markNow()
    var listener: IUnityAdsTokenListener? = null

    override suspend fun invoke(listener: IUnityAdsTokenListener?) {
        this.listener = listener
        tokenStart()

        if (listener == null) {
            tokenFailure(awaitedInit = false, reason = REASON_LISTENER_NULL)
            return
        }

        when (getInitializationState()) {
            INITIALIZED -> fetchToken(awaitedInit = false)
            NOT_INITIALIZED, FAILED -> tokenFailure(awaitedInit = false, reason = REASON_NOT_INITIALIZED)
            INITIALIZING -> {
                when (awaitInitialization(GET_TOKEN_TIMEOUT_MS)) {
                    INITIALIZED -> fetchToken(true)
                    FAILED -> tokenFailure(awaitedInit = true, reason = REASON_NOT_INITIALIZED)
                    else -> tokenFailure(awaitedInit = true, reason = REASON_TIMEOUT)
                }
            }
        }
    }

    private fun fetchToken(awaitedInit: Boolean) {
        var reason: String? = null
        var reasonDebug: String? = null
        val token = try {
            getHeaderBiddingToken()
        } catch (e: Exception) {
            reason = SendDiagnosticEvent.REASON_UNCAUGHT_EXCEPTION
            reasonDebug = e.retrieveUnityCrashValue()
            null
        }
        if (token == null) {
            tokenFailure(awaitedInit, reason, reasonDebug)
        } else {
            tokenSuccess(awaitedInit, token)
        }
    }

    private fun tokenSuccess(awaitedInit: Boolean, token: String) {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_SUCCESS,
            value = startTime.elapsedMillis(),
            tags = mapOf(
                SYNC to false.toString(),
                STATE to getInitializationState().toString(),
                AWAITED_INIT to awaitedInit.toString(),
            )
        )
        wrapCustomerListener { listener?.onUnityAdsTokenReady(token) }
    }

    private fun tokenFailure(awaitedInit: Boolean, reason: String?, reasonDebug: String? = null) {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_FAILURE,
            value = startTime.elapsedMillis(),
            tags = buildMap {
                put(SYNC, false.toString())
                put(STATE, getInitializationState().toString())
                put(AWAITED_INIT, awaitedInit.toString())
                reason?.let { put(REASON, reason) }
                reasonDebug?.let { put(REASON_DEBUG, reasonDebug) }
            })
        wrapCustomerListener { listener?.onUnityAdsTokenReady(null) }
    }

    private fun tokenStart() {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_STARTED,
            tags = mapOf(
                SYNC to false.toString(),
                STATE to getInitializationState().toString()
            )
        )
    }
}