package com.unity3d.ads.core.data.repository

import android.content.Context
import android.webkit.WebView
import com.google.protobuf.ByteString
import com.iab.omid.library.unity3d.adsession.AdSession
import com.iab.omid.library.unity3d.adsession.CreativeType
import com.iab.omid.library.unity3d.adsession.ImpressionType
import com.iab.omid.library.unity3d.adsession.Owner
import com.iab.omid.library.unity3d.adsession.Partner
import com.unity3d.ads.core.data.manager.OmidManager
import com.unity3d.ads.core.data.model.OMResult
import com.unity3d.ads.core.data.model.OMData
import com.unity3d.ads.core.data.model.OmidOptions
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.extensions.toISO8859String
import com.unity3d.services.UnityAdsConstants.ClientInfo.SDK_VERSION_NAME
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_PARTNER_NAME
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_PARTNER_VERSION
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.withContext

class AndroidOpenMeasurementRepository(
    private val mainDispatcher: CoroutineDispatcher,
    private val omidManager: OmidManager
) : OpenMeasurementRepository {
    private val partner = Partner.createPartner(OM_PARTNER_NAME, SDK_VERSION_NAME)
    private val activeSessions = MutableStateFlow(mapOf<String, AdSession>())
    private val _isOMActive = MutableStateFlow(value = false)

    override val omData: OMData
        get() = buildOmData()

    override var isOMActive: Boolean
        get() = _isOMActive.value
        set(value) = _isOMActive.update { value }

    override suspend fun activateOM(context: Context): OMResult = withContext(mainDispatcher) {
        if (isOMActive) {
            OMResult.Failure(reason = SendDiagnosticEvent.REASON_OM_ALREADY_ACTIVE)
        }

        try {
            omidManager.activate(context)
            isOMActive = omidManager.isActive
            if (isOMActive) {
                OMResult.Success
            } else {
               OMResult.Failure(reason = SendDiagnosticEvent.OM_ACTIVATE_FAILURE)
            }
        } catch (t: Throwable) {
           OMResult.Failure(
               reason = SendDiagnosticEvent.REASON_UNCAUGHT_EXCEPTION,
               reasonDebug = t.message.toString()
            )
        }
    }

    override suspend fun startSession(
        opportunityId: ByteString,
        webView: WebView?,
        options: OmidOptions
    ): OMResult = withContext(mainDispatcher) {
        try {
            if (!isOMActive) {
                OMResult.Failure(reason = SendDiagnosticEvent.REASON_OM_NOT_ACTIVE)
            }

            if (activeSessions.value.containsKey(opportunityId.toISO8859String())) {
               OMResult.Failure(reason = SendDiagnosticEvent.REASON_OM_SESSION_ALREADY_EXISTS)
            }

            val adSessionConfiguration = omidManager.createAdSessionConfiguration(
                options.creativeType ?: CreativeType.DEFINED_BY_JAVASCRIPT,
                options.impressionType ?: ImpressionType.DEFINED_BY_JAVASCRIPT,
                options.impressionOwner ?: Owner.JAVASCRIPT,
                options.videoEventsOwner ?: Owner.JAVASCRIPT,
                options.isolateVerificationScripts
            )

            val context = omidManager.createHtmlAdSessionContext(partner, webView, null, options.customReferenceData)
            val adSession = omidManager.createAdSession(adSessionConfiguration, context)
            adSession.registerAdView(webView)
            addSession(opportunityId, adSession)

            OMResult.Success
        } catch (t: Throwable) {
            OMResult.Failure(
                reason = SendDiagnosticEvent.REASON_UNCAUGHT_EXCEPTION,
                reasonDebug = t.message.toString()
            )
        }
    }

    /**
     * NOTE:
     *  Ending an OMID ad session sends a message to the verification scripts running inside the
     *  webview supplied by the integration. So that the verification scripts have enough time to
     *  handle the sessionFinish event, the integration must maintain a strong reference to the webview
     *  for at least 1.0 seconds after ending the session. If an error occurs, it will throw an exception.
     */
    override suspend fun finishSession(opportunityId: ByteString): OMResult = withContext(mainDispatcher) {
        if (!isOMActive) {
            OMResult.Failure(reason = SendDiagnosticEvent.REASON_OM_NOT_ACTIVE)
        }

        val session = getSession(opportunityId) ?: return@withContext OMResult.Failure(reason = SendDiagnosticEvent.REASON_OM_SESSION_NOT_FOUND)
        session.finish()
        removeSession(opportunityId)

        OMResult.Success
    }

    override suspend fun impressionOccurred(
        opportunityId: ByteString,
        signalLoaded: Boolean
    ): OMResult = withContext(mainDispatcher) {
        val session = getSession(opportunityId) ?: return@withContext OMResult.Failure(reason = SendDiagnosticEvent.REASON_OM_SESSION_NOT_FOUND)
        val adEvents = omidManager.createAdEvents(session)

        if (signalLoaded) {
            adEvents.loaded()
        }

        adEvents.impressionOccurred()
        OMResult.Success
    }

    private fun addSession(opportunityId: ByteString, adSession: AdSession) {
        activeSessions.value += opportunityId.toISO8859String() to adSession
    }

    private fun removeSession(opportunityId: ByteString) {
        activeSessions.value -= opportunityId.toISO8859String()
    }

    private fun getSession(opportunityId: ByteString): AdSession? {
        return activeSessions.value[opportunityId.toISO8859String()]
    }

    private fun buildOmData(): OMData {
        return OMData(
            version = omidManager.version,
            partnerName = OM_PARTNER_NAME,
            partnerVersion = OM_PARTNER_VERSION,
            activeSessions = activeSessions.value
        )
    }
}