/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.extensions;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.AccessController;
import java.security.NoSuchAlgorithmException;
import java.security.Permission;
import java.security.PermissionCollection;
import java.security.Permissions;
import java.security.Policy;
import java.security.URIParameter;
import java.security.UnresolvedPermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.SpecialPermission;
import org.elasticsearch.bootstrap.JarHell;
import org.elasticsearch.cli.EnvironmentAwareCommand;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xpack.XPackPlugin;
import org.elasticsearch.xpack.extensions.XPackExtensionInfo;
import org.elasticsearch.xpack.extensions.XPackExtensionSecurity;

final class InstallXPackExtensionCommand
extends EnvironmentAwareCommand {
    private final OptionSpec<Void> batchOption;
    private final OptionSpec<String> arguments;

    InstallXPackExtensionCommand() {
        super("Install an extension");
        this.batchOption = this.parser.acceptsAll(Arrays.asList("b", "batch"), "Enable batch mode explicitly, automatic confirmation of security permission");
        this.arguments = this.parser.nonOptions("extension id");
    }

    @Override
    protected void execute(Terminal terminal, OptionSet options, Environment env) throws Exception {
        List<String> args = this.arguments.values(options);
        if (args.size() != 1) {
            throw new UserException(64, "Must supply a single extension id argument");
        }
        String extensionURL = args.get(0);
        boolean isBatch = options.has(this.batchOption) || System.console() == null;
        this.execute(terminal, extensionURL, isBatch, env);
    }

    void execute(Terminal terminal, String extensionId, boolean isBatch, Environment env) throws Exception {
        if (!Files.exists(XPackPlugin.resolveXPackExtensionsFile(env), new LinkOption[0])) {
            terminal.println("xpack extensions directory [" + XPackPlugin.resolveXPackExtensionsFile(env) + "] does not exist. Creating...");
            Files.createDirectories(XPackPlugin.resolveXPackExtensionsFile(env), new FileAttribute[0]);
        }
        Path extensionZip = this.download(terminal, extensionId, env.tmpFile());
        Path extractedZip = this.unzip(extensionZip, XPackPlugin.resolveXPackExtensionsFile(env));
        this.install(terminal, extractedZip, env, isBatch);
    }

    private Path download(Terminal terminal, String extensionURL, Path tmpDir) throws Exception {
        terminal.println("-> Downloading " + URLDecoder.decode(extensionURL, "UTF-8"));
        URL url = new URL(extensionURL);
        Path zip = Files.createTempFile(tmpDir, null, ".zip", new FileAttribute[0]);
        try (InputStream in = url.openStream();){
            Files.copy(in, zip, StandardCopyOption.REPLACE_EXISTING);
        }
        return zip;
    }

    private Path unzip(Path zip, Path extensionDir) throws IOException, UserException {
        Path target = Files.createTempDirectory(extensionDir, ".installing-", new FileAttribute[0]);
        Files.createDirectories(target, new FileAttribute[0]);
        try (ZipInputStream zipInput = new ZipInputStream(Files.newInputStream(zip, new OpenOption[0]));){
            ZipEntry entry;
            byte[] buffer = new byte[8192];
            while ((entry = zipInput.getNextEntry()) != null) {
                Path targetFile = target.resolve(entry.getName());
                Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
                if (!entry.isDirectory()) {
                    try (OutputStream out = Files.newOutputStream(targetFile, new OpenOption[0]);){
                        int len;
                        while ((len = zipInput.read(buffer)) >= 0) {
                            out.write(buffer, 0, len);
                        }
                    }
                }
                zipInput.closeEntry();
            }
        }
        Files.delete(zip);
        return target;
    }

    private XPackExtensionInfo verify(Terminal terminal, Path extensionRoot, Environment env, boolean isBatch) throws Exception {
        XPackExtensionInfo info = XPackExtensionInfo.readFromProperties(extensionRoot);
        terminal.println(Terminal.Verbosity.VERBOSE, info.toString());
        this.jarHellCheck(extensionRoot);
        Path policy = extensionRoot.resolve("x-pack-extension-security.policy");
        if (Files.exists(policy, new LinkOption[0])) {
            InstallXPackExtensionCommand.readPolicy(policy, terminal, env, isBatch);
        }
        return info;
    }

    private void jarHellCheck(Path candidate) throws Exception {
        Path[] extensionJars;
        ArrayList<URL> jars = new ArrayList<URL>();
        jars.addAll(Arrays.asList(JarHell.parseClassPath()));
        for (Path jar : extensionJars = FileSystemUtils.files(candidate, "*.jar")) {
            jars.add(jar.toUri().toURL());
        }
        JarHell.checkJarHell(jars.toArray(new URL[jars.size()]));
    }

    private void install(Terminal terminal, Path tmpRoot, Environment env, boolean isBatch) throws Exception {
        ArrayList<Path> deleteOnFailure = new ArrayList<Path>();
        deleteOnFailure.add(tmpRoot);
        try {
            XPackExtensionInfo info = this.verify(terminal, tmpRoot, env, isBatch);
            Path destination = XPackPlugin.resolveXPackExtensionsFile(env).resolve(info.getName());
            if (Files.exists(destination, new LinkOption[0])) {
                throw new UserException(64, "extension directory " + destination.toAbsolutePath() + " already exists. To update the extension, uninstall it first using 'remove " + info.getName() + "' command");
            }
            Files.move(tmpRoot, destination, StandardCopyOption.ATOMIC_MOVE);
            terminal.println("-> Installed " + info.getName());
        }
        catch (Exception installProblem) {
            try {
                IOUtils.rm(deleteOnFailure.toArray(new Path[0]));
            }
            catch (IOException exceptionWhileRemovingFiles) {
                installProblem.addSuppressed(exceptionWhileRemovingFiles);
            }
            throw installProblem;
        }
    }

    static String formatPermission(Permission permission) {
        StringBuilder sb = new StringBuilder();
        String clazz = null;
        clazz = permission instanceof UnresolvedPermission ? ((UnresolvedPermission)permission).getUnresolvedType() : permission.getClass().getName();
        sb.append(clazz);
        String name = null;
        name = permission instanceof UnresolvedPermission ? ((UnresolvedPermission)permission).getUnresolvedName() : permission.getName();
        if (name != null && name.length() > 0) {
            sb.append(' ');
            sb.append(name);
        }
        String actions = null;
        actions = permission instanceof UnresolvedPermission ? ((UnresolvedPermission)permission).getUnresolvedActions() : permission.getActions();
        if (actions != null && actions.length() > 0) {
            sb.append(' ');
            sb.append(actions);
        }
        return sb.toString();
    }

    static PermissionCollection parsePermissions(Path file, Path tmpDir) throws IOException {
        Path emptyPolicyFile = Files.createTempFile(tmpDir, "empty", "tmp", new FileAttribute[0]);
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            sm.checkPermission(new SpecialPermission());
        }
        Policy emptyPolicy = AccessController.doPrivileged(() -> {
            try {
                return Policy.getInstance("JavaPolicy", new URIParameter(emptyPolicyFile.toUri()));
            }
            catch (NoSuchAlgorithmException e) {
                throw new RuntimeException(e);
            }
        });
        IOUtils.rm(emptyPolicyFile);
        Policy policy = AccessController.doPrivileged(() -> {
            try {
                return Policy.getInstance("JavaPolicy", new URIParameter(file.toUri()));
            }
            catch (NoSuchAlgorithmException e) {
                throw new RuntimeException(e);
            }
        });
        PermissionCollection permissions = policy.getPermissions(XPackExtensionSecurity.class.getProtectionDomain());
        if (permissions == Policy.UNSUPPORTED_EMPTY_COLLECTION) {
            throw new UnsupportedOperationException("JavaPolicy implementation does not support retrieving permissions");
        }
        Permissions actualPermissions = new Permissions();
        for (Permission permission : Collections.list(permissions.elements())) {
            if (emptyPolicy.implies(XPackExtensionSecurity.class.getProtectionDomain(), permission)) continue;
            ((PermissionCollection)actualPermissions).add(permission);
        }
        actualPermissions.setReadOnly();
        return actualPermissions;
    }

    static void readPolicy(Path file, Terminal terminal, Environment environment, boolean batch) throws IOException {
        PermissionCollection permissions = InstallXPackExtensionCommand.parsePermissions(file, environment.tmpFile());
        ArrayList<Permission> requested = Collections.list(permissions.elements());
        if (requested.isEmpty()) {
            terminal.println(Terminal.Verbosity.VERBOSE, "extension has a policy file with no additional permissions");
            return;
        }
        Collections.sort(requested, new Comparator<Permission>(){

            @Override
            public int compare(Permission o1, Permission o2) {
                int cmp = o1.getClass().getName().compareTo(o2.getClass().getName());
                if (cmp == 0) {
                    String name1 = o1.getName();
                    String name2 = o2.getName();
                    if (name1 == null) {
                        name1 = "";
                    }
                    if (name2 == null) {
                        name2 = "";
                    }
                    if ((cmp = name1.compareTo(name2)) == 0) {
                        String actions1 = o1.getActions();
                        String actions2 = o2.getActions();
                        if (actions1 == null) {
                            actions1 = "";
                        }
                        if (actions2 == null) {
                            actions2 = "";
                        }
                        cmp = actions1.compareTo(actions2);
                    }
                }
                return cmp;
            }
        });
        terminal.println(Terminal.Verbosity.NORMAL, "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
        terminal.println(Terminal.Verbosity.NORMAL, "@     WARNING: x-pack extension requires additional permissions     @");
        terminal.println(Terminal.Verbosity.NORMAL, "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
        for (Permission permission : requested) {
            terminal.println(Terminal.Verbosity.NORMAL, "* " + InstallXPackExtensionCommand.formatPermission(permission));
        }
        terminal.println(Terminal.Verbosity.NORMAL, "See http://docs.oracle.com/javase/8/docs/technotes/guides/security/permissions.html");
        terminal.println(Terminal.Verbosity.NORMAL, "for descriptions of what these permissions allow and the associated risks.");
        if (!batch) {
            terminal.println(Terminal.Verbosity.NORMAL, "");
            String text = terminal.readText("Continue with installation? [y/N]");
            if (!text.equalsIgnoreCase("y")) {
                throw new RuntimeException("installation aborted by user");
            }
        }
    }
}

