package com.twilio.video;

import android.os.Handler;
import org.jetbrains.annotations.NotNull;
import tvi.webrtc.VideoFrame;

/**
 * VideoSink implements webrtc.VideoSink interface. Currently it is used for detecting changes in
 * captured frame's parameters such as changes in width, height, and rotation. When these changes
 * are detected, it notifies the VideoCapturerObserver.
 */
public class VideoSink implements tvi.webrtc.VideoSink {

    private static final Logger logger = Logger.getLogger(VideoSink.class);
    private VideoCapturerObserver videoCapturerObserver;
    // Stores the last captured width, height, and rotation
    private int previousWidth;
    private int previousHeight;
    private int previousRotation;
    private Handler capturerChangesHandler;
    private final Runnable notifyCapturerChanges =
            new Runnable() {
                @Override
                public void run() {
                    if (videoCapturerObserver != null) {
                        videoCapturerObserver.onCaptureParametersChanged();
                    }
                }
            };

    public VideoSink(@NotNull VideoCapturerObserver videoCapturerObserver) {
        Preconditions.checkNotNull(
                videoCapturerObserver, "Video Capturer Observer must not be null");
        this.videoCapturerObserver = videoCapturerObserver;
        capturerChangesHandler = Util.createCallbackHandler();
    }

    public void resetVideoCapturerObserver() {
        this.videoCapturerObserver = null;
    }

    @Override
    public void onFrame(VideoFrame videoFrame) {
        videoFrame.retain();
        final int width = videoFrame.getBuffer().getWidth();
        final int height = videoFrame.getBuffer().getHeight();
        final int rotation = videoFrame.getRotation();
        if ((previousHeight != height)
                || (previousWidth != width)
                || (previousRotation != rotation)) {
            logger.d(
                    "Observed change in frame width from "
                            + previousWidth
                            + " to "
                            + width
                            + " or change in frame height from "
                            + previousHeight
                            + " to "
                            + height
                            + " or change in frame rotation from "
                            + previousRotation
                            + " to "
                            + rotation);
            if (capturerChangesHandler != null) {
                capturerChangesHandler.post(notifyCapturerChanges);
            } else {
                logger.e("Ignoring capturer change events. Handler not available.");
            }
        }
        previousHeight = height;
        previousWidth = width;
        previousRotation = rotation;
        videoFrame.release();
    }
}
