package com.twilio.video;

import androidx.annotation.NonNull;

/**
 * A local data track publication represents a {@link LocalDataTrack} that has been shared to a
 * {@link Room}.
 */
public class LocalDataTrackPublication implements DataTrackPublication {
    private final String sid;
    private final LocalDataTrack localDataTrack;
    private TrackPriority priority;
    private long nativeLocalDataTrackPublicationContext;

    LocalDataTrackPublication(
            @NonNull String sid,
            @NonNull LocalDataTrack localDataTrack,
            @NonNull TrackPriority priority,
            long nativeLocalDataTrackPublicationContext) {
        Preconditions.checkNotNull(sid, "SID must not be null");
        Preconditions.checkArgument(!sid.isEmpty(), "SID must not be empty");
        Preconditions.checkNotNull(localDataTrack, "Local data track must not be null");
        Preconditions.checkNotNull(priority, "priority should not be null");
        this.sid = sid;
        this.localDataTrack = localDataTrack;
        this.priority = priority;
        this.nativeLocalDataTrackPublicationContext = nativeLocalDataTrackPublicationContext;
    }

    /**
     * Returns the local data track's server identifier. This value uniquely identifies the local
     * data track within the scope of a {@link Room}.
     */
    @NonNull
    @Override
    public String getTrackSid() {
        return sid;
    }

    /**
     * Returns the name of the local data track. An empty string is returned if no name was
     * specified.
     */
    @NonNull
    @Override
    public String getTrackName() {
        return localDataTrack.getName();
    }

    /** Check if local data track is enabled. */
    @Override
    public boolean isTrackEnabled() {
        return localDataTrack.isEnabled();
    }

    /**
     * Unlike {@link DataTrackPublication} or {@link RemoteDataTrackPublication} this {@link
     * DataTrack} is guaranteed to be non null.
     *
     * @return Returns the base data track object of the published local data track.
     */
    @Override
    public DataTrack getDataTrack() {
        return localDataTrack;
    }

    /** Returns the published local data track. */
    @NonNull
    public LocalDataTrack getLocalDataTrack() {
        return localDataTrack;
    }

    /**
     * Returns the {@link TrackPriority} of the published local data track.
     *
     * @deprecated Retrieving track priority of local data tracks is deprecated since {@link
     *     TrackPriority} is a deprecated feature.
     */
    @NonNull
    @Deprecated
    public TrackPriority getPriority() {
        return priority;
    }

    /**
     * Update the publisher's {@link TrackPriority} for this {@link LocalDataTrack} after it has
     * been published. This method is a no-op if the {@link LocalDataTrack} has been unpublished, or
     * the {@link LocalParticipant} has disconnected from the {@link Room}.
     *
     * @param priority The {@link TrackPriority} to be set.
     * @deprecated Setting track priority of local data tracks is deprecated since {@link
     *     TrackPriority} is a deprecated feature.
     */
    @Deprecated
    public void setPriority(TrackPriority priority) {
        this.priority = priority;
        if (!isReleased()) {
            nativeSetPriority(nativeLocalDataTrackPublicationContext, priority);
        }
    }

    /*
     * Releases native memory owned by local audio data publication.
     */
    synchronized void release() {
        if (!isReleased()) {
            nativeRelease(nativeLocalDataTrackPublicationContext);
            nativeLocalDataTrackPublicationContext = 0;
        }
    }

    synchronized boolean isReleased() {
        return nativeLocalDataTrackPublicationContext == 0;
    }

    private native void nativeSetPriority(
            long nativeLocalDataTrackPublicationContext, TrackPriority priority);

    private native void nativeRelease(long nativeLocalDataTrackPublicationContext);
}
