package com.twilio.video;

import androidx.annotation.NonNull;

/**
 * Defines audio and video encoding parameters. Maximum bitrate is specified as Transport
 * Independent Application Specific Maximum (TIAS) bitrate <a
 * href="https://tools.ietf.org/html/rfc3890">RFC3890</a> in kilobits per second (kbps) excluding
 * IP/UDP/TCP headers. **Note:** These encoding parameters are applied for each peer connection,
 * each track and each simulcast layer.
 *
 * <p>- For group Rooms, there is a single peer connection to Twilio Media Server, but for
 * peer-to-peer Rooms, there is a separate peer connection for each participant. I.e., if you set
 * maximum video bitrate to 1 Mbps and you have two {@link RemoteParticipant}`s in the Room, the
 * effective video send bitrate may be up to 2 Mbps.
 *
 * <p>- If you are publishing multiple video tracks (e.g. camera and screen share), each track
 * receives the maximum bitrate specified. I.e. if you set maximum video bitrate to 1 Mbps and you
 * publish both a camera and a screen share track, the effective video send bitrate may be up to 2
 * Mbps.
 *
 * <p>- If simulcast is used for video tracks, each simulcast layer receives the maximum bitrate
 * specified. I.e. if your set the maximum bitrate to 1 Mbps and you publish a simulcast video
 * track, the effective video send bitrate may be up to 3 Mbps if there are three simulcast layers.
 *
 * <p>Encoding parameters can be updated any time, using {@link
 * LocalParticipant#setEncodingParameters(EncodingParameters)}.
 */
public class EncodingParameters {
    /**
     * Maximum audio send bitrate in kilobits per second (kbps). Zero indicates the WebRTC default
     * value, which is codec dependent. The maximum bitrate for <a
     * href="http://opus-codec.org/">Opus</a> is 510 Kbps.
     */
    public final int maxAudioBitrate;

    /**
     * Maximum video send bitrate in kilobits per second (kbps). Zero indicates the WebRTC default
     * value, which is 2,000 Kbps.
     */
    public final int maxVideoBitrate;

    public EncodingParameters(int maxAudioBitrate, int maxVideoBitrate) {
        this.maxAudioBitrate = maxAudioBitrate;
        this.maxVideoBitrate = maxVideoBitrate;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        EncodingParameters that = (EncodingParameters) o;

        return maxAudioBitrate == that.maxAudioBitrate && maxVideoBitrate == that.maxVideoBitrate;
    }

    @Override
    public int hashCode() {
        int result = maxAudioBitrate;
        result = 31 * result + maxVideoBitrate;
        return result;
    }

    @NonNull
    @Override
    public String toString() {
        return "EncodingParameters{"
                + "maxAudioBitrate="
                + maxAudioBitrate
                + ", maxVideoBitrate="
                + maxVideoBitrate
                + '}';
    }
}
