package com.twilio.video;

import android.content.Context;
import android.content.res.TypedArray;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import tvi.webrtc.RendererCommon;
import tvi.webrtc.SurfaceViewRenderer;
import tvi.webrtc.VideoFrame;

/**
 * A VideoView renders frames from a {@link VideoTrack}. This class is an extension of {@link
 * android.view.SurfaceView}, so it can be placed in your XML view hierarchy.
 */
public class VideoView extends SurfaceViewRenderer {
    private static final Logger logger = Logger.getLogger(VideoView.class);

    // Used to ensure that our renderer has a means to post to main thread for renderer events
    private final Handler uiThreadHandler = new Handler(Looper.getMainLooper());
    @VisibleForTesting VideoSinkHintsProducer videoSinkHintsProducer;
    private OnAttachStateChangeListener onAttachStateChangeListener =
            new OnAttachStateChangeListener() {
                @Override
                public void onViewDetachedFromWindow(View v) {
                    if (videoSinkHintsProducer != null) {
                        videoSinkHintsProducer.produceSinkHints(View.GONE);
                    }
                }

                @Override
                public void onViewAttachedToWindow(View v) {
                    if (videoSinkHintsProducer != null) {
                        videoSinkHintsProducer.produceSinkHints(View.VISIBLE);
                    }
                }
            };

    private final RendererCommon.RendererEvents internalEventListener =
            new RendererCommon.RendererEvents() {
                @Override
                public void onFirstFrameRendered() {
                    refreshRenderer();
                    if (listener != null) {
                        listener.onFirstFrameRendered();
                    }
                }

                @Override
                public void onFrameResolutionChanged(
                        int videoWidth, int videoHeight, int rotation) {
                    refreshRenderer();
                    if (listener != null) {
                        listener.onFrameResolutionChanged(videoWidth, videoHeight, rotation);
                    }
                }
            };

    private boolean mirror;
    private boolean overlaySurface;
    private VideoScaleType videoScaleType;
    private RendererCommon.RendererEvents listener;
    private EglBaseProvider eglBaseProvider;

    public VideoView(@NonNull Context context) {
        this(context, null);
    }

    public VideoView(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        TypedArray a =
                context.getTheme().obtainStyledAttributes(attrs, R.styleable.VideoView, 0, 0);

        try {
            mirror = a.getBoolean(R.styleable.VideoView_tviMirror, false);
            videoScaleType =
                    VideoScaleType.fromInt(a.getInteger(R.styleable.VideoView_tviScaleType, 0));
            overlaySurface = a.getBoolean(R.styleable.VideoView_tviOverlaySurface, false);
        } finally {
            a.recycle();
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        // Do not setup the renderer when using developer tools to avoid EGL14 runtime exceptions
        if (!isInEditMode()) {
            eglBaseProvider = EglBaseProvider.instance(this);
            setupRenderer();
        }
    }

    @Override
    public void onFrame(VideoFrame frame) {
        frame.retain();
        super.onFrame(frame);
        frame.release();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.release();
        this.eglBaseProvider.release(this);
        super.onDetachedFromWindow();
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);
        if (videoSinkHintsProducer != null) {
            videoSinkHintsProducer.produceSinkHints(visibility);
        }
    }

    @Override
    protected void onSizeChanged(int width, int height, int oldWidth, int oldHeight) {
        super.onSizeChanged(width, height, oldWidth, oldHeight);
        if (videoSinkHintsProducer != null) {
            int widthDp = UtilKt.convertPixelsToDp(getContext(), width);
            int heightDp = UtilKt.convertPixelsToDp(getContext(), height);
            videoSinkHintsProducer.produceSinkHints(widthDp, heightDp);
        }
    }

    /** Returns whether or not this view is mirroring video. */
    public boolean getMirror() {
        return mirror;
    }

    /** Sets whether or not the rendered video should be mirrored. */
    public void setMirror(boolean mirror) {
        if (this.mirror != mirror) {
            this.mirror = mirror;
            super.setMirror(mirror);
            refreshRenderer();
        }
    }

    /** Returns the current {@link VideoScaleType}. */
    @NonNull
    public VideoScaleType getVideoScaleType() {
        return videoScaleType;
    }

    /**
     * Sets the current scale type to specified value and updates the video.
     *
     * <p><b>Note</b>: The scale type will only be applied to dimensions defined as {@link
     * android.view.ViewGroup.LayoutParams#WRAP_CONTENT} or a custom value. Setting a width or
     * height to {@link android.view.ViewGroup.LayoutParams#MATCH_PARENT} results in the video being
     * scaled to fill the maximum value of the dimension.
     */
    public void setVideoScaleType(@NonNull VideoScaleType videoScaleType) {
        ViewGroup.LayoutParams layoutParams = getLayoutParams();

        // Log warning if scale type may not be respected in certain dimensions
        if (layoutParams != null
                && (layoutParams.width == ViewGroup.LayoutParams.MATCH_PARENT
                        || layoutParams.height == ViewGroup.LayoutParams.MATCH_PARENT)) {
            VideoScaleType widthScaleType =
                    (layoutParams.width == ViewGroup.LayoutParams.MATCH_PARENT)
                            ? (VideoScaleType.ASPECT_FILL)
                            : (videoScaleType);
            VideoScaleType heightScaleType =
                    (layoutParams.height == ViewGroup.LayoutParams.MATCH_PARENT)
                            ? (VideoScaleType.ASPECT_FILL)
                            : (videoScaleType);

            logger.w(
                    String.format(
                            "Scale type may not be applied as expected because "
                                    + "video view uses MATCH_PARENT. Scaling will be applied as "
                                    + "follows: width=%s, height=%s",
                            widthScaleType.name(), heightScaleType.name()));
        }

        this.videoScaleType = videoScaleType;
        setScalingType(convertToWebRtcScaleType(videoScaleType));
        refreshRenderer();
    }

    /** Sets listener of rendering events. */
    public void setListener(@Nullable RendererCommon.RendererEvents listener) {
        this.listener = listener;
    }

    /**
     * Controls placement of the video render relative to other surface.
     *
     * @param overlaySurface if true, video renderer is placed on top of another video renderer in
     *     the window (but still behind window itself).
     */
    public void applyZOrder(boolean overlaySurface) {
        this.overlaySurface = overlaySurface;
        setZOrderMediaOverlay(overlaySurface);
    }

    void setupVideoSinkHintsProducer(
            VideoSinkHintsConsumer videoSinkHintsConsumer, long sinkHintsId) {
        this.videoSinkHintsProducer =
                new VideoSinkHintsProducer(
                        getContext().getResources().getDisplayMetrics().densityDpi,
                        videoSinkHintsConsumer,
                        sinkHintsId,
                        getVisibility(),
                        UtilKt.convertPixelsToDp(getContext(), getWidth()),
                        UtilKt.convertPixelsToDp(getContext(), getHeight()));
        addOnAttachStateChangeListener(onAttachStateChangeListener);
    }

    void removeVideoSinkHintsProducer() {
        videoSinkHintsProducer = null;
        removeOnAttachStateChangeListener(onAttachStateChangeListener);
    }

    long getSinkHintsId() {
        return this.videoSinkHintsProducer.getSinkHintsId();
    }

    private void setupRenderer() {
        init(eglBaseProvider.getRootEglBase().getEglBaseContext(), internalEventListener);
        setScalingType(convertToWebRtcScaleType(videoScaleType));
        setZOrderMediaOverlay(overlaySurface);
        super.setMirror(mirror);
        refreshRenderer();
    }

    private void refreshRenderer() {
        uiThreadHandler.post(this::requestLayout);
    }

    private RendererCommon.ScalingType convertToWebRtcScaleType(VideoScaleType videoScaleType) {
        switch (videoScaleType) {
            case ASPECT_FILL:
                return RendererCommon.ScalingType.SCALE_ASPECT_FILL;
            case ASPECT_BALANCED:
                return RendererCommon.ScalingType.SCALE_ASPECT_BALANCED;
            default:
                return RendererCommon.ScalingType.SCALE_ASPECT_FIT;
        }
    }
}
