package com.twilio.video;

import android.os.Handler;
import android.util.Pair;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import tvi.webrtc.RTCStatsCollectorCallback;
import tvi.webrtc.RTCStatsReport;

/**
 * RtcStatsAdapter represents the Java layer object that manages the delivery of RtcStatsReports to
 * the {@link RtcStatsListener}s. It does the following: 1. It holds references to the {@link
 * RtcStatsListener}s until the time the RtcStatsReports are ready for delivery. 2. It receives the
 * {@link RTCStatsReport} from the JNI layer. 3. It delivers the {@link RTCStatsReport} to the
 * {@link RtcStatsListener}s when the RtcStatsReports for all the Peer Connections have been
 * received.
 */
public class RtcStatsAdapter implements RTCStatsCollectorCallback {
    private static final Logger logger = Logger.getLogger(RtcStatsAdapter.class);

    @Override
    public synchronized void onStatsDelivered(RTCStatsReport statsReport) {
        // Add the stats report to the collection
        rtcStatsReports.add(statsReport);
    }

    // Raise the onRtcStats callback on all queued listeners now that the stats reports have been
    // received for all peer connections.
    public synchronized void raiseOnRtcStats() {
        logger.d("Calling onRtcStats on all RtcStatsListeners");
        while (true) {
            final Pair<Handler, RtcStatsListener> statsPair = rtcStatsListenersQueue.poll();
            if (statsPair == null) {
                break;
            }
            List<RTCStatsReport> rtcStatsReportsCopy = new ArrayList<>(rtcStatsReports);
            statsPair.first.post(() -> statsPair.second.onRtcStats(rtcStatsReportsCopy));
        }
        rtcStatsReports.clear();
    }

    public synchronized void addRtcStatsListener(RtcStatsListener rtcStatsListener) {
        rtcStatsListenersQueue.offer(new Pair<>(Util.createCallbackHandler(), rtcStatsListener));
    }

    public synchronized void cleanup() {
        for (final Pair<Handler, RtcStatsListener> listenerPair : rtcStatsListenersQueue) {
            logger.d("cleanup: Calling onRtcStats with an empty RTC stats report.");
            listenerPair.first.post(
                    () -> listenerPair.second.onRtcStats(new ArrayList<RTCStatsReport>()));
        }
        rtcStatsListenersQueue.clear();
        rtcStatsReports.clear();
    }

    private List<RTCStatsReport> rtcStatsReports = new ArrayList<>();
    private Queue<Pair<Handler, RtcStatsListener>> rtcStatsListenersQueue =
            new ConcurrentLinkedQueue<>();
}
