package com.twilio.video;

import androidx.annotation.NonNull;
import java.nio.ByteBuffer;

/**
 * AudioDevice interface allows developers to inject custom audio device capturer and audio device
 * renderer of audio by replacing the default device used by the SDK.
 */
public interface AudioDevice extends AudioDeviceCapturer, AudioDeviceRenderer {
    /**
     * This method should be called any time the capturing or rendering format changes. If both
     * format change simultaneously its fine to call this method just once. As a result, the media
     * engine will read the new audio format, stop capturing and/or rendering by calling {@link
     * AudioDeviceCapturer#onStopCapturing()} and/or {@link AudioDeviceRenderer#onStopRendering()}
     * and then call {@link AudioDeviceCapturer#onStartCapturing(AudioDeviceContext)} and/or {@link
     * AudioDeviceRenderer#onStartRendering(AudioDeviceContext)}.
     *
     * @param audioDeviceContext The AudioDevice context. You should use the same context provided
     *     in {@link AudioDeviceCapturer#onStartCapturing(AudioDeviceContext)} and/or {@link
     *     AudioDeviceRenderer#onStartRendering(AudioDeviceContext)} here.
     */
    static void audioDeviceFormatChanged(@NonNull AudioDeviceContext audioDeviceContext) {
        Preconditions.checkNotNull(audioDeviceContext, "audioDeviceContext must not be null");
        AudioDeviceProxy audioDeviceProxy = (AudioDeviceProxy) audioDeviceContext;
        audioDeviceProxy.formatChanged();
    }

    /**
     * This method needs to be called by {@link AudioDeviceCapturer} implementation to provide
     * captured data to the media engine. When {@link
     * AudioDeviceCapturer#onStartCapturing(AudioDeviceContext)} callback is raised, call this
     * method in a loop to provide captured data to media engine.
     *
     * @param audioDeviceContext The AudioDevice context. You should use the same context provided
     *     in {@link AudioDeviceCapturer#onStartCapturing(AudioDeviceContext)} and/or {@link
     *     AudioDeviceRenderer#onStartRendering(AudioDeviceContext)} here.
     * @param audioSample Audio data you wish to deliver.
     */
    static void audioDeviceWriteCaptureData(
            @NonNull AudioDeviceContext audioDeviceContext, @NonNull ByteBuffer audioSample) {
        Preconditions.checkNotNull(audioDeviceContext, "audioDeviceContext must not be null");
        Preconditions.checkNotNull(audioSample, "audioSample must not be null");

        AudioDeviceProxy audioDeviceProxy = (AudioDeviceProxy) audioDeviceContext;
        audioDeviceProxy.writeCaptureData(audioSample);
    }

    /**
     * This method needs to be called by {@link AudioDeviceRenderer} to pull renderable audio data
     * from the media engine.
     *
     * @param audioDeviceContext The context reference. You should use the same context provided in
     *     {@link AudioDeviceRenderer#onStartRendering(AudioDeviceContext)} here.
     * @param audioSample A reference to a buffer where the signed 16-bit LPCM audio data will be
     *     copied into.
     */
    static void audioDeviceReadRenderData(
            @NonNull AudioDeviceContext audioDeviceContext, @NonNull ByteBuffer audioSample) {
        Preconditions.checkNotNull(audioDeviceContext, "audioDeviceContext must not be null");
        Preconditions.checkNotNull(audioSample, "runnable must not be null");

        AudioDeviceProxy audioDeviceProxy = (AudioDeviceProxy) audioDeviceContext;
        audioDeviceProxy.readRenderData(audioSample);
    }

    /**
     * A utility method to execute a runnable on the media engine's worker thread asynchronously.
     *
     * <p>AudioDevice invokes the callbacks on the media engine's worker thread. If you wish to
     * initialize, start or stop the audio device from your application code, to avoid the
     * thread-safety problems, you should execute code from the media engine's worker thread.
     *
     * @param audioDeviceContext The AudioDevice context. You should use the same context provided
     *     in {@link AudioDeviceRenderer#onStartRendering(AudioDeviceContext)} or {@link
     *     AudioDeviceCapturer#onStartCapturing(AudioDeviceContext)} here.
     * @param runnable A block which will be executed on the media engine's worker thread.
     */
    static void audioDeviceExecuteWorkerBlock(
            @NonNull AudioDeviceContext audioDeviceContext, @NonNull Runnable runnable) {
        Preconditions.checkNotNull(audioDeviceContext, "audioDeviceContext must not be null");
        Preconditions.checkNotNull(runnable, "runnable must not be null");

        AudioDeviceProxy audioDeviceProxy = (AudioDeviceProxy) audioDeviceContext;
        audioDeviceProxy.executeWorkerBlock(runnable);
    }
}
