package com.twilio.video;

import android.util.DisplayMetrics;
import android.view.View;

class VideoSinkHintsProducer {
    private final float scaleFactor;
    private VideoSinkHintsConsumer videoSinkHintsConsumer;
    private final long sinkHintsId;
    private SinkHints cachedSinkHints;

    VideoSinkHintsProducer(
            int densityDpi,
            VideoSinkHintsConsumer videoSinkHintsConsumer,
            long sinkHintsId,
            int viewVisibility,
            int viewWidth,
            int viewHeight) {
        this.videoSinkHintsConsumer = videoSinkHintsConsumer;
        this.sinkHintsId = sinkHintsId;

        scaleFactor = densityDpi <= DisplayMetrics.DENSITY_MEDIUM ? 1f : 1.5f;

        // Produce initial sink hints
        this.cachedSinkHints = new SinkHints(sinkHintsId, null, null);
        produceSinkHints(viewVisibility);
        produceSinkHints(viewWidth, viewHeight);
    }

    void produceSinkHints(int viewVisibility) {
        if (videoSinkHintsConsumer != null
                && videoSinkHintsConsumer.isClientTrackSwitchAutoModeEnabled()) {
            SinkHints sinkHints =
                    new SinkHints(
                            sinkHintsId,
                            viewVisibility == View.VISIBLE,
                            cachedSinkHints.getVideoContentPreferences());
            cachedSinkHints = sinkHints;
            videoSinkHintsConsumer.consumeSinkHints(sinkHints);
        }
    }

    void produceSinkHints(int width, int height) {
        if (videoSinkHintsConsumer != null
                && videoSinkHintsConsumer.isVideoContentPreferencesAutoModeEnabled()) {
            VideoContentPreferences videoContentPreferences =
                    getScaledVideoContentPreferences(width, height);
            SinkHints sinkHints =
                    new SinkHints(
                            sinkHintsId, cachedSinkHints.isEnabled(), videoContentPreferences);
            cachedSinkHints = sinkHints;
            videoSinkHintsConsumer.consumeSinkHints(sinkHints);
        }
    }

    long getSinkHintsId() {
        return sinkHintsId;
    }

    private VideoContentPreferences getScaledVideoContentPreferences(int width, int height) {
        int scaledWidth = Math.round(width * scaleFactor);
        int scaledHeight = Math.round(height * scaleFactor);
        VideoDimensions videoDimensions = new VideoDimensions(scaledWidth, scaledHeight);
        return new VideoContentPreferences(videoDimensions);
    }
}
