package com.twilio.video;

import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.media.projection.MediaProjection;
import android.os.Handler;
import android.util.DisplayMetrics;
import android.view.Surface;
import android.view.WindowManager;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import tvi.webrtc.CapturerObserver;
import tvi.webrtc.ScreenCapturerAndroid;
import tvi.webrtc.SurfaceTextureHelper;

/**
 * The ScreenCapturer class is used to provide video frames for a {@link LocalVideoTrack} from a
 * device's screen. The frames are provided via the {@link MediaProjection} api. This capturer is
 * only compatible with {@link android.os.Build.VERSION_CODES#LOLLIPOP} or higher.
 *
 * <p>This class represents an implementation of a {@link VideoCapturer} interface. Although public,
 * these methods are not meant to be invoked directly.
 *
 * <p><b>Note</b>: This capturer can be reused, but cannot be shared across multiple {@link
 * LocalVideoTrack}s simultaneously.
 */
@TargetApi(21)
public class ScreenCapturer implements VideoCapturer {
    private static final Logger logger = Logger.getLogger(ScreenCapturer.class);

    private static final int SCREENCAPTURE_FRAME_RATE = 30;

    private boolean firstFrameReported;
    private int orientation;

    private final Context context;
    private final Intent screenCaptureIntentData;
    private final Listener screenCapturerListener;
    private final int screenCaptureIntentResult;
    private final Handler listenerHandler;

    private CapturerObserver capturerObserver;
    private ScreenCapturerAndroid webRtcScreenCapturer;

    private VideoDimensions videoDimensions;

    private final MediaProjection.Callback mediaProjectionCallback =
            new MediaProjection.Callback() {
                @Override
                public void onStop() {
                    super.onStop();
                    logger.d("media projection stopped");
                }
            };

    private final tvi.webrtc.CapturerObserver observerAdapter =
            new tvi.webrtc.CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean success) {
                    logger.d("screen capturer started");
                    if (!success) {
                        if (screenCapturerListener != null) {
                            listenerHandler.post(
                                    () ->
                                            screenCapturerListener.onScreenCaptureError(
                                                    "Failed to start screen capturer"));
                        }
                    }
                    capturerObserver.onCapturerStarted(success);
                }

                @Override
                public void onCapturerStopped() {
                    logger.d("screen capturer stopped");
                }

                @Override
                public void onFrameCaptured(tvi.webrtc.VideoFrame videoFrame) {
                    if (!firstFrameReported) {
                        if (screenCapturerListener != null) {
                            listenerHandler.post(screenCapturerListener::onFirstFrameAvailable);
                        }
                        firstFrameReported = true;
                    }
                    tvi.webrtc.VideoFrame.Buffer buffer = videoFrame.getBuffer();
                    videoDimensions = new VideoDimensions(buffer.getWidth(), buffer.getHeight());
                    int currentOrientation = getDeviceOrientation();

                    capturerObserver.onFrameCaptured(videoFrame);

                    if (updateCaptureDimensions(orientation, currentOrientation)) {
                        // Swap width and height capture dimensions due to orientation update
                        logger.d("Swapping width and height of frame due to orientation");
                        orientation = currentOrientation;
                        webRtcScreenCapturer.changeCaptureFormat(
                                buffer.getHeight(), buffer.getWidth(), SCREENCAPTURE_FRAME_RATE);
                    }
                }
            };

    /** Interface that provides events and errors related to {@link ScreenCapturer}. */
    public interface Listener {
        /**
         * Reports an error that occurred in {@link ScreenCapturer}.
         *
         * @param errorDescription description of the error that occurred.
         */
        void onScreenCaptureError(@NonNull String errorDescription);

        /** Indicates when the first frame has been captured from the screen. */
        void onFirstFrameAvailable();
    }

    public ScreenCapturer(
            @NonNull Context context,
            int screenCaptureIntentResult,
            @NonNull Intent screenCaptureIntentData,
            @Nullable Listener screenCapturerListener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(screenCaptureIntentData, "intent must not be null");

        this.context = context;
        this.screenCaptureIntentData = screenCaptureIntentData;
        this.screenCaptureIntentResult = screenCaptureIntentResult;
        this.screenCapturerListener = screenCapturerListener;
        this.listenerHandler = Util.createCallbackHandler();
        this.orientation = getDeviceOrientation();
    }

    @Override
    public VideoFormat getCaptureFormat() {
        DisplayMetrics displayMetrics = new DisplayMetrics();
        WindowManager windowManager =
                (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        windowManager.getDefaultDisplay().getRealMetrics(displayMetrics);
        VideoDimensions screenDimensions =
                new VideoDimensions(displayMetrics.widthPixels, displayMetrics.heightPixels);

        return new VideoFormat(screenDimensions, SCREENCAPTURE_FRAME_RATE);
    }

    /** Indicates that the screen capturer is a screencast. */
    @Override
    public boolean isScreencast() {
        return true;
    }

    @Override
    public void initialize(
            @NonNull SurfaceTextureHelper surfaceTextureHelper,
            @NonNull Context context,
            @NonNull CapturerObserver capturerObserver) {
        this.capturerObserver = capturerObserver;
        this.webRtcScreenCapturer =
                new ScreenCapturerAndroid(screenCaptureIntentData, mediaProjectionCallback);
        webRtcScreenCapturer.initialize(surfaceTextureHelper, context, observerAdapter);
    }

    @Override
    public void startCapture(int width, int height, int framerate) {
        this.firstFrameReported = false;
        if (screenCaptureIntentResult != Activity.RESULT_OK) {
            if (screenCapturerListener != null) {
                listenerHandler.post(
                        () ->
                                screenCapturerListener.onScreenCaptureError(
                                        "MediaProjection permissions must be granted to start ScreenCapturer"));
            }
            observerAdapter.onCapturerStarted(false);
            return;
        }
        webRtcScreenCapturer.startCapture(width, height, framerate);
    }

    /**
     * Stops all frames being captured. {@link MediaProjection} should be available for use upon
     * completion.
     *
     * <p><b>Note</b>: This method is not meant to be invoked directly.
     */
    @Override
    public void stopCapture() {
        logger.d("stopCapture");
        if (webRtcScreenCapturer != null) {
            webRtcScreenCapturer.stopCapture();
        }
        logger.d("stopCapture done");
    }

    @Override
    public void dispose() {
        if (webRtcScreenCapturer != null) {
            webRtcScreenCapturer.dispose();
            webRtcScreenCapturer = null;
        }
    }

    private int getDeviceOrientation() {
        int orientation = 0;

        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        switch (wm.getDefaultDisplay().getRotation()) {
            case Surface.ROTATION_90:
                orientation = 90;
                break;
            case Surface.ROTATION_180:
                orientation = 180;
                break;
            case Surface.ROTATION_270:
                orientation = 270;
                break;
            case Surface.ROTATION_0:
                break;
        }
        return orientation;
    }

    private boolean updateCaptureDimensions(int currentOrientation, int orientation) {
        if (currentOrientation == orientation
                || Math.abs(currentOrientation - orientation) == 180) {
            logger.d("No orientation change detected");
            return false;
        } else {
            logger.d("Orientation change detected");
            return true;
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    VideoDimensions getVideoDimensions() {
        return videoDimensions;
    }
}
