package com.twilio.video;

import android.content.Context;
import android.content.res.AssetManager;
import androidx.annotation.VisibleForTesting;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Properties;

/* The global configuration is to manage settings that need to be set before the sdk is
 * instantiated. A use-case for this would be for enabling/disabling features associated Video
 * encoding in the field. If it exists, it loads a properties file from application's assets
 * directory. This properties file, one can enable/disable internal features that should not be set
 * at run time. The location of the properties file can be changed by setting/changing the system
 * property (com.twilio.video.Video.configPath).
 *
 * The current properties defined (mainly from webrtc) are...
 * tvi.webrtc.DefaultVideoEncoderFactory.enableHWEncoding = {true/false}
 * tvi.webrtc.DefaultVideoEncoderFactory.enableHWVP8Encoding = {true/false}
 * tvi.webrtc.DefaultVideoEncoderFactory.enableHWVP9Encoding = {true/false}
 * tvi.webrtc.DefaultVideoEncoderFactory.enableHWH264Encoding = {true/false}
 * tvi.webrtc.HardwareVideoEncoderFactory.enableH264HighProfile = {true/false}
 * tvi.webrtc.HardwareVideoEncoderFactory.enableIntelVP8Encoder = {true/false}
 * tvi.webrtc.HardwareVideoEncoderFactory.enableH264HuaweiSupport = {true/false}
 */

class GlobalConfiguration {
    public static final String SYSPROP_GLOBAL_CONFIG_FILE = "com.twilio.video.Video.configPath";
    public static final String DEFAULT_CONFIG_FILE = "twilio-video-config.properties";

    private final Properties _configuration = new Properties(System.getProperties());
    private static GlobalConfiguration _instance = null;
    private static final Logger _logger = Logger.getLogger(GlobalConfiguration.class);

    public static GlobalConfiguration getInstance(Context context) {
        if (null == _instance) {
            _instance = new GlobalConfiguration(context);
        }
        return _instance;
    }

    public Properties getConfig() {
        return _configuration;
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    static void release() {
        _instance = null;
    }

    private GlobalConfiguration(Context context) {
        AssetManager assetManager = context.getAssets();
        try {
            String configFilePath =
                    System.getProperty(SYSPROP_GLOBAL_CONFIG_FILE, DEFAULT_CONFIG_FILE);
            InputStream inputStream = assetManager.open(configFilePath);
            _configuration.load(inputStream);
            inputStream.close();
        } catch (FileNotFoundException fnfe) {
            _logger.i("There is no Global Configuration override file: " + DEFAULT_CONFIG_FILE);
        } catch (Exception e) {
            _logger.e(
                    "Global Configuration override file: "
                            + SYSPROP_GLOBAL_CONFIG_FILE
                            + " failed to load. Error:"
                            + e.getMessage());
            e.printStackTrace();
        }
    }
}
