package com.twilio.video;

import static java.lang.annotation.RetentionPolicy.SOURCE;

import android.Manifest;
import android.annotation.TargetApi;
import android.content.Context;
import android.hardware.camera2.CaptureRequest;
import android.os.Build;
import android.os.Handler;
import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.lang.annotation.Retention;
import java.util.concurrent.atomic.AtomicBoolean;
import tvi.webrtc.Camera2Enumerator;
import tvi.webrtc.Camera2Session;
import tvi.webrtc.CameraVideoCapturer;
import tvi.webrtc.CapturerObserver;
import tvi.webrtc.SurfaceTextureHelper;

/**
 * The Camera2Capturer class is used to provide video frames for a {@link LocalVideoTrack} from the
 * provided {@link #cameraId}. The frames are provided via a {@link
 * android.hardware.camera2.CameraCaptureSession}. Camera2Capturer must be run on devices {@link
 * android.os.Build.VERSION_CODES#LOLLIPOP} or higher.
 *
 * <p>This class represents an implementation of a {@link VideoCapturer} interface. Although public,
 * these methods are not meant to be invoked directly.
 *
 * <p>This capturer can be reused, but cannot be shared across multiple {@link LocalVideoTrack}s
 * simultaneously. It is possible to construct multiple instances with different camera IDs, but
 * there are device specific limitations on how many camera2 sessions can be open.
 */
@TargetApi(21)
public class Camera2Capturer implements VideoCapturer {
    private static final String CAMERA_SWITCH_PENDING_ERROR_MESSAGE =
            "Camera switch already in progress.";
    private static final Logger logger = Logger.getLogger(Camera2Capturer.class);

    private final Object stateLock = new Object();
    private Camera2Capturer.State state = Camera2Capturer.State.IDLE;

    private final Context applicationContext;
    private final Camera2Enumerator camera2Enumerator;
    private final AtomicBoolean captureRequestUpdatePending = new AtomicBoolean(false);
    private CaptureRequestUpdater captureRequestUpdater;
    @Nullable private final Listener listener;
    private final Handler handler;
    private String cameraId;
    private CapturerObserver capturerObserver;
    private tvi.webrtc.Camera2Capturer webrtcCamera2Capturer;
    private tvi.webrtc.Camera2Session camera2Session;
    private String pendingCameraId;

    private final CameraVideoCapturer.CameraEventsHandler cameraEventsHandler =
            new CameraVideoCapturer.CameraEventsHandler() {
                @Override
                public void onCameraError(final String errorMessage) {
                    reportError(new Exception(Exception.UNKNOWN, errorMessage));
                }

                @Override
                public void onCameraDisconnected() {}

                @Override
                public void onCameraFreezed(final String errorMessage) {
                    reportError(new Exception(Exception.CAMERA_FROZE, errorMessage));
                }

                @Override
                public void onCameraOpening(String s) {}

                @Override
                public void onFirstFrameAvailable() {
                    setSession();
                    handler.post(
                            () -> {
                                if (listener != null) {
                                    listener.onFirstFrameAvailable();
                                }
                            });
                }

                @Override
                public void onCameraClosed() {}
            };
    private final tvi.webrtc.CapturerObserver observerAdapter =
            new tvi.webrtc.CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean success) {
                    if (success) {
                        synchronized (stateLock) {
                            if (state == State.STARTING) {
                                setSession();
                                state = Camera2Capturer.State.RUNNING;

                                /*
                                 * The user requested a capture request update while the capturer was not
                                 * running. Apply the capture request update now that the capturer
                                 * is running.
                                 */
                                if (captureRequestUpdater != null) {
                                    updateCaptureRequestOnCameraThread(captureRequestUpdater);
                                    captureRequestUpdater = null;
                                }
                            } else {
                                logger.w("Attempted to transition from " + state + " to RUNNING");
                            }
                        }
                    } else {
                        logger.e("Failed to start capturer");
                    }
                    capturerObserver.onCapturerStarted(success);
                }

                @Override
                public void onCapturerStopped() {
                    synchronized (stateLock) {
                        state = Camera2Capturer.State.IDLE;
                    }
                    capturerObserver.onCapturerStopped();
                }

                @Override
                public void onFrameCaptured(tvi.webrtc.VideoFrame videoFrame) {
                    capturerObserver.onFrameCaptured(videoFrame);
                }
            };

    private final CameraVideoCapturer.CameraSwitchHandler cameraSwitchHandler =
            new CameraVideoCapturer.CameraSwitchHandler() {
                @Override
                public void onCameraSwitchDone(boolean isFrontCamera) {
                    synchronized (Camera2Capturer.this) {
                        cameraId = pendingCameraId;
                        pendingCameraId = null;
                    }
                    handler.post(
                            () -> {
                                if (listener != null) {
                                    listener.onCameraSwitched(cameraId);
                                }
                            });
                }

                @Override
                public void onCameraSwitchError(final String errorMessage) {
                    logger.e("Failed to switch to camera with ID: " + pendingCameraId);
                    synchronized (Camera2Capturer.this) {
                        pendingCameraId = null;
                    }
                    reportError(new Exception(Exception.CAMERA_SWITCH_FAILED, errorMessage));
                }
            };

    /**
     * Indicates if Camera2Capturer is compatible with device.
     *
     * <p>This method checks that all the following conditions are true: <br>
     *
     * <ol>
     *   <li>The device API level is at least {@link android.os.Build.VERSION_CODES#LOLLIPOP}.
     *   <li>All device cameras have hardware support level greater than {@link
     *       android.hardware.camera2.CameraCharacteristics#INFO_SUPPORTED_HARDWARE_LEVEL_LEGACY}.
     * </ol>
     *
     * <br>
     * For more details on supported hardware levels see the <a
     * href="https://developer.android.com/reference/android/hardware/camera2/CameraCharacteristics.html#INFO_SUPPORTED_HARDWARE_LEVEL">Android
     * documentation</a>.
     *
     * @param context application context.
     * @return true if device supports Camera2Capturer and false if not.
     */
    public static boolean isSupported(@NonNull Context context) {
        Preconditions.checkNotNull(context, "Context must not be null");
        return Camera2Enumerator.isSupported(context);
    }

    /**
     * Constructs a Camera2Capturer instance.
     *
     * @param context application context
     * @param cameraId unique identifier of the camera device to open that must be specified in
     *     {@link android.hardware.camera2.CameraManager#getCameraIdList()}
     */
    public Camera2Capturer(@NonNull Context context, @NonNull String cameraId) {
        this(context, cameraId, null, Util.createCallbackHandler());
    }

    /**
     * Constructs a Camera2Capturer instance.
     *
     * @param context application context
     * @param cameraId unique identifier of the camera device to open that must be specified in
     *     {@link android.hardware.camera2.CameraManager#getCameraIdList()}
     * @param listener an optional listener of camera2 capturer events
     */
    public Camera2Capturer(
            @NonNull Context context, @NonNull String cameraId, @Nullable Listener listener) {
        this(context, cameraId, listener, Util.createCallbackHandler());
    }

    /*
     * Package scope constructor that allows passing in a mocked handler for unit tests.
     */

    @VisibleForTesting
    Camera2Capturer(
            @NonNull Context context,
            @NonNull String cameraId,
            @Nullable Listener listener,
            @NonNull Handler handler) {
        Preconditions.checkState(
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP,
                "Camera2Capturer unavailable for " + Build.VERSION.SDK_INT);
        Preconditions.checkNotNull(context, "Context must not be null");
        Preconditions.checkState(
                isSupported(context), "Camera2Capturer is not supported on this device");
        Preconditions.checkNotNull(cameraId, "Camera ID must not be null");
        Preconditions.checkArgument(!cameraId.isEmpty(), "Camera ID must not be empty");
        this.applicationContext = context.getApplicationContext();
        this.camera2Enumerator = new Camera2Enumerator(applicationContext);
        this.cameraId = cameraId;
        this.listener = listener;
        this.handler = handler;
    }
    /** Indicates that the camera2 capturer is not a screen cast. */
    @Override
    public boolean isScreencast() {
        return false;
    }

    @Override
    public void initialize(
            @NonNull SurfaceTextureHelper surfaceTextureHelper,
            @NonNull Context context,
            @NonNull CapturerObserver capturerObserver) {
        this.capturerObserver = capturerObserver;
        this.webrtcCamera2Capturer =
                (tvi.webrtc.Camera2Capturer)
                        camera2Enumerator.createCapturer(cameraId, cameraEventsHandler);
        this.webrtcCamera2Capturer.initialize(surfaceTextureHelper, context, observerAdapter);
    }

    @Override
    public void startCapture(int width, int height, int framerate) {
        checkCapturerState();
        synchronized (stateLock) {
            if (state == State.IDLE || state == State.STOPPING) {
                logger.i("startCapture");
                state = Camera2Capturer.State.STARTING;
                this.webrtcCamera2Capturer.startCapture(width, height, framerate);
            } else if (state == State.STARTING || state == State.RUNNING) {
                logger.w("attempted to start capturing when capturer is starting or running");
            }
        }
    }

    /**
     * Stops all frames being captured.
     *
     * <p><b>Note</b>: This method is not meant to be invoked directly.
     */
    @Override
    public void stopCapture() {
        if (webrtcCamera2Capturer != null) {
            synchronized (stateLock) {
                state = State.STOPPING;
            }
            webrtcCamera2Capturer.stopCapture();
        }
    }

    @Override
    public void dispose() {
        if (webrtcCamera2Capturer != null) {
            webrtcCamera2Capturer.dispose();
            webrtcCamera2Capturer = null;
        }
    }

    /** Returns the currently set camera ID. */
    @NonNull
    public synchronized String getCameraId() {
        return cameraId;
    }

    /**
     * Switches the current {@link #cameraId}.
     *
     * @param newCameraId the new camera id.
     */
    public synchronized void switchCamera(@NonNull final String newCameraId) {
        Preconditions.checkNotNull(newCameraId, "Camera ID must not be null");
        Preconditions.checkArgument(!newCameraId.isEmpty(), "Camera ID must not be empty");
        Preconditions.checkArgument(
                !newCameraId.equals(cameraId),
                "Camera ID must be different " + "from current camera ID");
        Preconditions.checkArgument(
                Camera2Utils.cameraIdSupported(applicationContext, newCameraId),
                "Camera ID %s is not supported or could not be validated",
                newCameraId);
        synchronized (stateLock) {
            if (state != Camera2Capturer.State.IDLE) {
                if (pendingCameraId == null) {
                    pendingCameraId = newCameraId;
                    webrtcCamera2Capturer.switchCamera(cameraSwitchHandler, newCameraId);
                } else {
                    handler.post(
                            () ->
                                    reportError(
                                            new Exception(
                                                    Exception.CAMERA_SWITCH_FAILED,
                                                    CAMERA_SWITCH_PENDING_ERROR_MESSAGE)));
                }
            } else {
                cameraId = newCameraId;
                if (listener != null) {
                    listener.onCameraSwitched(cameraId);
                }
            }
        }
    }

    /**
     * Schedules a capture request update.
     *
     * <p>A {@link CaptureRequest.Builder} optimal for capturing video for streaming will be
     * provided for modification to the provided {@link CaptureRequestUpdater}. Modifications to the
     * {@link CaptureRequest.Builder} will be applied to the {@link
     * android.hardware.camera2.CameraCaptureSession} after the invocation of {@link
     * CaptureRequestUpdater#apply(CaptureRequest.Builder)}. {@link
     * #updateCaptureRequest(CaptureRequestUpdater)} can be invoked while capturing frames or not.
     * If the {@link Camera2Capturer} is not capturing the {@link CaptureRequestUpdater} will be
     * called when capturer resumes.
     *
     * <p>The following snippet demonstrates how to turn on the flash of a camera while capturing.
     *
     * <pre><code>
     *     // Create the camera2 capturer
     *     Camera2Capturer camera2Capturer = new Camera2Capturer(context, cameraId, camera2Listener);
     *
     *     // Start the camera2 capturer
     *     LocalVideoTrack camera2VideoTrack = LocalVideoTrack.create(context, true, camera2Capturer);
     *
     *     // Schedule the capture request update
     *     camera2Capturer.updateCaptureRequest(new CaptureRequestUpdater() {
     *         {@literal @}Override
     *         public void apply(@NonNull CaptureRequest.Builder captureRequestBuilder) {
     *             captureRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
     *         }
     *     }
     * </code></pre>
     *
     * <p>The camera2 capturer will raise an {@link Exception} to the provided {@link Listener} with
     * the code {@link Exception#CAPTURE_REQUEST_UPDATE_FAILED} if a successfully scheduled capture
     * request update failed to be applied.
     *
     * @param captureRequestUpdater capture request updater that receives the capture request
     *     builder that can be modified.
     * @return true if the update was scheduled or false if an update is pending or could not be
     *     scheduled.
     */
    public synchronized boolean updateCaptureRequest(
            @NonNull final CaptureRequestUpdater captureRequestUpdater) {
        synchronized (stateLock) {
            if (state == State.RUNNING) {
                if (!captureRequestUpdatePending.get()) {
                    captureRequestUpdatePending.set(true);
                    return webrtcCamera2Capturer
                            .getHandler()
                            .post(() -> updateCaptureRequestOnCameraThread(captureRequestUpdater));
                } else {
                    logger.w("Parameters will not be applied with parameter update pending");
                    return false;
                }
            } else {
                logger.i(
                        "Camera2Capturer is not running. The CaptureRequest update will be applied when "
                                + "it is resumed");
                this.captureRequestUpdater = captureRequestUpdater;

                return true;
            }
        }
    }

    private void setSession() {
        logger.d("Camera capture session configured");
        camera2Session = (Camera2Session) webrtcCamera2Capturer.getCameraSession();
    }

    private void checkCapturerState() {
        Preconditions.checkState(
                Util.permissionGranted(applicationContext, Manifest.permission.CAMERA),
                "CAMERA permission must be granted to create video" + "track with Camera2Capturer");
        Preconditions.checkState(
                Camera2Utils.cameraIdSupported(applicationContext, cameraId),
                "Camera ID %s is not supported or could not be validated",
                cameraId);
    }

    private void reportError(Exception camera2Exception) {
        logger.e(camera2Exception.getMessage(), camera2Exception);
        handler.post(
                () -> {
                    if (listener != null) {
                        listener.onError(camera2Exception);
                    }
                });
    }

    private void updateCaptureRequestOnCameraThread(
            final @NonNull CaptureRequestUpdater captureRequestUpdater) {
        Preconditions.checkNotNull(captureRequestUpdater, "captureRequestUpdate must not be null");
        synchronized (stateLock) {
            if (state == State.RUNNING) {
                logger.i("Applying custom capture request");
                CaptureRequest.Builder captureRequestBuilder =
                        camera2Session.configureCaptureRequestBuilder();

                if (captureRequestBuilder == null) {
                    reportError(
                            new Exception(
                                    Exception.CAPTURE_REQUEST_UPDATE_FAILED,
                                    "Failed to create capture request builder"));
                    captureRequestUpdatePending.set(false);
                    return;
                }
                captureRequestUpdater.apply(captureRequestBuilder);
                boolean succeeded =
                        camera2Session.setSessionRepeatingRequest(captureRequestBuilder);

                if (!succeeded) {
                    reportError(
                            new Exception(
                                    Exception.CAPTURE_REQUEST_UPDATE_FAILED,
                                    "Failed to set session repeating request"));
                    captureRequestUpdatePending.set(false);
                    return;
                }
            } else {
                logger.w(
                        "Attempted to update camera parameters while camera capturer is "
                                + "not running");
            }

            // Clear the parameter updating flag
            captureRequestUpdatePending.set(false);
        }
    }

    /** Camera2Capturer exception class. */
    public static class Exception extends TwilioException {
        @Retention(SOURCE)
        @IntDef({CAMERA_SWITCH_FAILED, CAMERA_FROZE, UNKNOWN, CAPTURE_REQUEST_UPDATE_FAILED})
        public @interface Code {}

        public static final int CAMERA_FROZE = 0;
        public static final int CAMERA_SWITCH_FAILED = 1;
        public static final int UNKNOWN = 2;
        public static final int CAPTURE_REQUEST_UPDATE_FAILED = 3;

        Exception(@Code int code, @NonNull String message, @Nullable String explanation) {
            super(code, message, explanation);
        }

        Exception(@Code int code, @NonNull String message) {
            this(code, message, null);
        }
    }

    /*
     * State definitions used to control interactions with the public API
     */
    private enum State {
        IDLE,
        STARTING,
        RUNNING,
        STOPPING
    }

    /** Interface that provides events and errors related to {@link Camera2Capturer}. */
    public interface Listener {
        /** Indicates when the first frame has been captured from the camera. */
        void onFirstFrameAvailable();

        /**
         * Notifies when a camera switch is complete.
         *
         * @param newCameraId the camera ID after camera switch is complete.
         */
        void onCameraSwitched(@NonNull String newCameraId);

        /**
         * Reports an error that occurred in {@link Camera2Capturer}.
         *
         * @param camera2CapturerException the code that describes the error that occurred.
         */
        void onError(@NonNull Exception camera2CapturerException);
    }
}
