/*
 * Copyright (C) 2018 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;

/**
 * Lossy audio coding format.
 *
 * @see <a href="https://en.wikipedia.org/wiki/Opus_(audio_format)">Opus</a>
 */
public class OpusCodec extends AudioCodec {
    @NonNull public static final String NAME = "opus";
    private final boolean useDtx;

    public OpusCodec() {
        this(true);
    }

    /**
     * A constructor to initialize the OpusCodec with non-default attributes.
     *
     * @param useDtx Discontinuous transmission reduces the bitrate for certain input signals, like
     *     periods of silence [RFC-7587 3.1.3](https://tools.ietf.org/html/rfc7587#section-3.1.3).
     *     The default is false. This setting applies to sender and receiver transmission.
     */
    public OpusCodec(boolean useDtx) {
        super(NAME);
        this.useDtx = useDtx;
    }

    /**
     * Reports whether DTX is enabled or not.
     *
     * @return true if DTX is enabled.
     */
    public boolean isDtxEnabled() {
        return useDtx;
    }
}
