/*
 * Copyright (C) 2017 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import tvi.webrtc.CapturerObserver;
import tvi.webrtc.SurfaceTextureHelper;
import tvi.webrtc.VideoSink;
import tvi.webrtc.VideoSource;

/**
 * A local video track that receives video frames from a {@link VideoCapturer} or {@link
 * tvi.webrtc.VideoCapturer}.
 *
 * <p>When a local video track is created, the SDK will call {@link
 * tvi.webrtc.VideoCapturer#initialize(SurfaceTextureHelper, Context, CapturerObserver)} and {@link
 * tvi.webrtc.VideoCapturer#startCapture(int, int, int)} on the calling thread before returning the
 * local video track instance. The capturer will be instructed to start capturing at the default
 * video format of 640x480 at 30 FPS unless any of the following conditions are true:
 *
 * <p>
 *
 * <ul>
 *   <li>The caller provides a specific {@link VideoFormat} when creating the local video track. In
 *       this case, the {@link tvi.webrtc.VideoCapturer} is responsible for capturing at the
 *       provided {@link VideoFormat}.
 *   <li>The capturer is an instance of {@link VideoCapturer} and no {@link VideoFormat} is provided
 *       when creating the local video track. In this case, the SDK will call {@link
 *       tvi.webrtc.VideoCapturer#startCapture(int, int, int)} with the result of {@link
 *       VideoCapturer#getCaptureFormat()}.
 * </ul>
 *
 * @see VideoCapturer
 */
public class LocalVideoTrack extends VideoTrack {
    private static final Logger logger = Logger.getLogger(LocalVideoTrack.class);
    static final VideoFormat DEFAULT_VIDEO_FORMAT =
            new VideoFormat(VideoDimensions.VGA_VIDEO_DIMENSIONS, 30);

    private long nativeLocalVideoTrackHandle;
    private final String nativeTrackHash;
    private final tvi.webrtc.VideoCapturer videoCapturer;
    private final VideoFormat videoFormat;
    private final MediaFactory mediaFactory;
    private final SurfaceTextureHelper surfaceTextureHelper;
    private final VideoSource videoSource;

    /**
     * Creates a local video track.
     *
     * <p>If the capturer is an instance of {@link VideoCapturer} then {@link
     * tvi.webrtc.VideoCapturer#startCapture(int, int, int)} will be called with the result of
     * {@link VideoCapturer#getCaptureFormat()}. This behavior allows {@link VideoCapturer}
     * implementations to provide a better default capture format. For example, the {@link
     * ScreenCapturer} provides a resolution derived from the device's screen dimensions.
     *
     * <p>If the capturer is an instance of {@link tvi.webrtc.VideoCapturer} then the default format
     * of 640x480 at 30 FPS will be used.
     *
     * @param context application context.
     * @param enabled initial state of video track.
     * @param videoCapturer capturer that provides video frames.
     * @return local video track if successfully added or {@code null} if video track could not be
     *     created.
     */
    @Nullable
    public static LocalVideoTrack create(
            @NonNull Context context,
            boolean enabled,
            @NonNull tvi.webrtc.VideoCapturer videoCapturer) {
        return create(context, enabled, videoCapturer, null, null);
    }

    /**
     * Creates a local video track.
     *
     * <p>The video capturer will be instructed to capture at the provided {@link VideoFormat}.
     *
     * @param context application context.
     * @param enabled initial state of video track.
     * @param videoCapturer capturer that provides video frames.
     * @param videoFormat format to capture frames.
     * @return local video track if successfully added or null if video track could not be created.
     */
    @Nullable
    public static LocalVideoTrack create(
            @NonNull Context context,
            boolean enabled,
            @NonNull tvi.webrtc.VideoCapturer videoCapturer,
            @Nullable VideoFormat videoFormat) {
        return create(context, enabled, videoCapturer, videoFormat, null);
    }

    /**
     * Creates a local video track.
     *
     * <p>If the capturer is an instance of {@link VideoCapturer} then {@link
     * tvi.webrtc.VideoCapturer#startCapture(int, int, int)} will be called with the result of
     * {@link VideoCapturer#getCaptureFormat()}. This behavior allows {@link VideoCapturer}
     * implementations to provide a better default capture format. For example, the {@link
     * ScreenCapturer} provides a resolution derived from the device's screen dimensions.
     *
     * <p>If the capturer is an instance of {@link tvi.webrtc.VideoCapturer} then the default format
     * of 640x480 at 30 FPS will be used.
     *
     * @param context application context.
     * @param enabled initial state of video track.
     * @param videoCapturer capturer that provides video frames.
     * @param name video track name.
     * @return local video track if successfully added or null if video track could not be created.
     */
    @Nullable
    public static LocalVideoTrack create(
            @NonNull Context context,
            boolean enabled,
            @NonNull tvi.webrtc.VideoCapturer videoCapturer,
            @Nullable String name) {
        return create(context, enabled, videoCapturer, null, name);
    }

    /**
     * Creates a local video track.
     *
     * <p>The video capturer will be instructed to capture at the provided {@link VideoFormat}.
     *
     * @param context application context.
     * @param enabled initial state of video track.
     * @param videoCapturer capturer that provides video frames.
     * @param videoFormat format to capture frames.
     * @param name video track name.
     * @return local video track if successfully added or null if video track could not be created.
     */
    @Nullable
    public static LocalVideoTrack create(
            @NonNull Context context,
            boolean enabled,
            @NonNull tvi.webrtc.VideoCapturer videoCapturer,
            @Nullable VideoFormat videoFormat,
            @Nullable String name) {
        Preconditions.checkNotNull(context, "Context must not be null");
        Preconditions.checkNotNull(videoCapturer, "VideoCapturer must not be null");

        // Use temporary media factory owner to create local video track
        Object temporaryMediaFactoryOwner = new Object();
        MediaFactory mediaFactory = MediaFactory.instance(temporaryMediaFactoryOwner, context);

        /*
         * If the developer has specified a format, then that format overrides
         * the format specified by VideoCapturer#getCaptureFormat.
         *
         * If the developer has not specified a format, then allow a VideoCapturer to specify
         * the format. If the developer or the VideoCapturer implementation does not specify a
         * format then the default capture format will be used.
         */
        if (videoFormat == null) {
            if (videoCapturer instanceof VideoCapturer) {
                videoFormat = ((VideoCapturer) videoCapturer).getCaptureFormat();
            } else {
                videoFormat = DEFAULT_VIDEO_FORMAT;
            }
        }

        LocalVideoTrack localVideoTrack =
                mediaFactory.createVideoTrack(context, enabled, videoCapturer, videoFormat, name);

        if (localVideoTrack == null) {
            logger.e("Failed to create local video track");
        } else {
            localVideoTrack.videoCapturer.initialize(
                    localVideoTrack.surfaceTextureHelper,
                    context,
                    localVideoTrack.videoSource.getCapturerObserver());
            localVideoTrack.videoCapturer.startCapture(
                    videoFormat.dimensions.width,
                    videoFormat.dimensions.height,
                    videoFormat.framerate);
        }

        // Local video track will obtain media factory instance in constructor so release ownership
        mediaFactory.release(temporaryMediaFactoryOwner);

        return localVideoTrack;
    }

    /** Returns the {@link tvi.webrtc.VideoCapturer} associated with this video track. */
    @NonNull
    public tvi.webrtc.VideoCapturer getVideoCapturer() {
        return videoCapturer;
    }

    /** Returns the video track's {@link VideoFormat}. */
    @NonNull
    public VideoFormat getVideoFormat() {
        return videoFormat;
    }

    /** Returns the {@link tvi.webrtc.VideoSource} associated with this video track. */
    @NonNull
    public VideoSource getVideoSource() {
        return videoSource;
    }

    @Override
    public synchronized void addSink(@NonNull VideoSink videoSink) {
        Preconditions.checkState(
                !isReleased(), "Cannot add sink to video track that has " + "been released");
        super.addSink(videoSink);
    }

    public synchronized void removeSink(@NonNull VideoSink videoSink) {
        Preconditions.checkState(
                !isReleased(), "Cannot remove sink from video track that has " + "been released");
        super.removeSink(videoSink);
    }

    /**
     * Check if the local video track is enabled.
     *
     * <p>When the value is false, blank video frames are sent. When the value is true, frames from
     * the video capturer are provided.
     *
     * @return true if the local video is enabled.
     */
    @Override
    public synchronized boolean isEnabled() {
        if (!isReleased()) {
            return nativeIsEnabled(nativeLocalVideoTrackHandle);
        } else {
            logger.e("Local video track is not enabled because it has been released");
            return false;
        }
    }

    /**
     * Returns the local video track name. A pseudo random string is returned if no track name was
     * specified.
     */
    @NonNull
    @Override
    public String getName() {
        return super.getName();
    }

    /**
     * Sets the state of the local video track. The results of this operation are signaled to other
     * Participants in the same Room. When a video track is disabled, blank frames are sent in place
     * of video frames from a video capturer.
     *
     * @param enabled the desired state of the local video track.
     */
    public synchronized void enable(boolean enabled) {
        if (!isReleased()) {
            nativeEnable(nativeLocalVideoTrackHandle, enabled);
        } else {
            logger.e("Cannot enable a local video track that has been removed");
        }
    }

    /** Releases native memory owned by video track. */
    public synchronized void release() {
        if (!isReleased()) {
            super.release();
            try {
                videoCapturer.stopCapture();
            } catch (InterruptedException e) {
                logger.e(e.getMessage(), e);
            }
            videoCapturer.dispose();
            videoSource.dispose();
            surfaceTextureHelper.dispose();
            nativeRelease(nativeLocalVideoTrackHandle);
            nativeLocalVideoTrackHandle = 0;
            mediaFactory.release(this);
        }
    }

    @AccessedByNative
    LocalVideoTrack(
            long nativeLocalVideoTrackHandle,
            boolean enabled,
            tvi.webrtc.VideoCapturer videoCapturer,
            VideoFormat videoFormat,
            tvi.webrtc.VideoTrack webrtcVideoTrack,
            String nativeTrackHash,
            String name,
            Context context,
            SurfaceTextureHelper surfaceTextureHelper,
            VideoSource videoSource) {
        super(webrtcVideoTrack, enabled, name);
        this.nativeTrackHash = nativeTrackHash;
        this.nativeLocalVideoTrackHandle = nativeLocalVideoTrackHandle;
        this.videoCapturer = videoCapturer;
        this.videoFormat = videoFormat;
        this.mediaFactory = MediaFactory.instance(this, context);
        this.surfaceTextureHelper = surfaceTextureHelper;
        this.videoSource = videoSource;
    }

    /*
     * Add and remove sink with wants are used in VideoCapturerTest to validate cases where
     * video capturer delegate needs to apply rotation to frames provided by a capturer. Applying
     * rotation is required for capturers when a VideoSinkInterface specifies with
     * rtc::VideoSinkWants.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    long addSinkWithWants(@NonNull VideoSink videoSink, boolean rotationApplied) {
        long nativeVideoSinkHandle =
                nativeAddSinkWithWants(nativeLocalVideoTrackHandle, rotationApplied);
        super.addSink(videoSink);

        return nativeVideoSinkHandle;
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    void removeSinkWithWants(long nativeVideoSinkHandle) {
        nativeRemoveSinkWithWants(nativeLocalVideoTrackHandle, nativeVideoSinkHandle);
    }

    boolean isReleased() {
        return nativeLocalVideoTrackHandle == 0;
    }

    /*
     * Called by LocalParticipant at JNI level to map twilio::media::LocalVideoTrack to
     * LocalVideoTrack.
     */
    @SuppressWarnings("unused")
    String getNativeTrackHash() {
        return nativeTrackHash;
    }

    /*
     * Called by LocalParticipant at JNI level
     */
    @SuppressWarnings("unused")
    @AccessedByNative
    synchronized long getNativeHandle() {
        return nativeLocalVideoTrackHandle;
    }

    private native boolean nativeIsEnabled(long nativeLocalVideoTrackHandle);

    private native void nativeEnable(long nativeLocalVideoTrackHandle, boolean enable);

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    private native long nativeAddSinkWithWants(
            long nativeLocalVideoTrackHandle, boolean rotationApplied);

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    private native void nativeRemoveSinkWithWants(
            long nativeLocalVideoTrackHandle, long nativeVideoSinkHandle);

    private native void nativeRelease(long nativeLocalVideoTrackHandle);
}
