package com.surveymonkey.surveymonkeyandroidsdk

import android.app.Activity
import android.content.Context
import android.os.Bundle
import androidx.fragment.app.FragmentActivity
import com.surveymonkey.surveymonkeyandroidsdk.SurveyDialogFragment
import com.surveymonkey.surveymonkeyandroidsdk.listener.DialogButtonClickListener
import com.surveymonkey.surveymonkeyandroidsdk.manager.SurveyManager
import com.surveymonkey.surveymonkeyandroidsdk.repository.SurveyRepositoryImpl
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMConstants
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMConstants.THREE_DAYS
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMConstants.THREE_MONTHS
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMConstants.THREE_WEEKS
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMNetworkUtils
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMNetworkUtils.buildURL

import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import org.json.JSONException
import org.json.JSONObject
import java.util.*

class SurveyMonkey :DialogButtonClickListener{
    private var mCollectorHash: String? = null
    private lateinit var mContext: Activity
    private var mSPageHTML: String? = null
    private val currentDate = Date().time

    lateinit var surveyManager: SurveyManager

    /**
     * Use the onStart method to present an AlertDialog prompt to the user encouraging them to take your in-app feedback survey.
     * By default, the user will be prompted to take a feedback survey 3 days after installing the app. If they decline to take the survey, they'll be prompted again in 3 weeks. Finally, if they consent to take the survey, they'll be prompted to take it again in 3 months.
     *
     * @param activity      The activity that will spawn the SMFeedbackActivity - **Note:** The spawning activity must implement onActivityResult to capture the respondent JSONObject or SMError object returned by SurveyMonkey
     * @param appName       The name of your app - used in the survey prompt dialog e.g. "Enjoying the SurveyMonkey app?"
     * @param requestCode   If greater than or equal to 0, this code will be returned in onActivityResult() when the activity exits.
     * @param collectorHash The collector hash that points to your Mobile SDK Collector - found on the Collector Details page of your chosen collector on com
     *
     * Once the user completes your survey, a JSONObject containing the user's response will be returned to the spawning activity in onActivityResult
     *
     * **Note:** If there are problems collecting the user's response, the SDK will return an SMError object with one of the domains and error codes listed in the SMError documentation
     * @see com.surveymonkey.newsmandroidsdk.utils.SMError
     */

    fun onStart(context: Activity, appName: String?, requestCode: Int, collectorHash: String, vararg customVariables: JSONObject?) {
        surveyManager = SurveyManager(SurveyRepositoryImpl())
        mContext = context
        onStart(context,requestCode, collectorHash,
            context.getString(R.string.sm_prompt_title_text, appName),
            context.getString(R.string.sm_prompt_message_text), THREE_DAYS, THREE_WEEKS,
            THREE_MONTHS,   *customVariables)
    }

    /**
     * Use the onStart method to present a customized AlertDialog prompt (at various intervals that you define) to the user encouraging them to take your in-app feedback survey.
     *
     * @param activity             The activity that will spawn the SMFeedbackActivity - **Note:** The spawning activity must implement onActivityResult to capture the respondent JSONObject or SMError object returned by SurveyMonkey
     * @param requestCode          If greater than or equal to 0, this code will be returned in onActivityResult() when the activity exits.
     * @param collectorHash        The collector hash that points to your Mobile SDK Collector - found on the Collector Details page of your chosen collector on com
     *
     * Once the user completes your survey, a JSONObject containing the user's response will be returned to the spawning activity in onActivityResult
     *
     * **Note:** If there are problems collecting the user's response, the SDK will return an SMError object with one of the domains and error codes listed in the SMError documentation
     * @param alertTitleText       The title of the survey prompt dialog displayed to the user
     * @param alertBodyText        The body of the survey prompt dialog displayed to the user
     * @param afterInstallInterval The amount of time (in milliseconds) to wait to first prompt the user to take your survey after they install your app
     * @param afterDeclineInterval The amount of time (in milliseconds) to wait to prompt the user to take your survey after they decline to take it
     * @param afterAcceptInterval  The amount of time (in milliseconds) to wait to prompt the user to take your survey after they consent to take it
     * @param customVariables (Optional) PLATINUM USERS ONLY: If you've set up custom variables in Create, you can pass a JSON object with the format {"key":"value",...} here.
     * @see com.surveymonkey.newsmandroidsdk.utils.SMError
     */
    private fun onStart(activity: Activity, requestCode: Int,
                        collectorHash: String,
                        alertTitleText: String?,
                        alertBodyText: String?, afterInstallInterval: Long, afterDeclineInterval: Long, afterAcceptInterval:Long,
                        vararg customVariables: JSONObject?
    ) {
        mContext = activity
        val prefs = mContext.getSharedPreferences(SMConstants.PREF_NAME, Context.MODE_PRIVATE)

        if (SMNetworkUtils.getConnectivityType(mContext) != SMNetworkUtils.ConnectivityType.NO_CONNECTION.value) {
            val promptDate = prefs.getLong(SMConstants.PROMPT_DATE, 0)
            if (promptDate == 0L) {
                prefs.edit().putLong(SMConstants.PROMPT_DATE, currentDate + afterInstallInterval).apply()
            } else if (promptDate < currentDate) {
                mCollectorHash = collectorHash
                runBlocking {
                    withContext(Dispatchers.IO){
                        handleServerResponse(surveyManager.getDataFromServer(collectorHash, *customVariables), alertTitleText, alertBodyText, requestCode, collectorHash,afterDeclineInterval, afterAcceptInterval, *customVariables)
                    }
                }

            }
        } else {
            prefs.edit().putLong(SMConstants.PROMPT_DATE, currentDate + afterInstallInterval).apply()
        }
    }

    private fun handleServerResponse(data: JSONObject?, alertTitleText: String?, alertBodyText: String?, requestCode: Int,
                                             collectorHash: String, afterDeclineInterval: Long, afterAcceptInterval: Long, vararg customVariables: JSONObject?){
        try {
            if (data != null) {
                val sdkData: JSONObject = data.getJSONObject(SMConstants.SURVEY_STATUS)
                mSPageHTML = data.getString(SMConstants.HTML)
                if (!sdkData.getBoolean(SMConstants.COLLECTOR_CLOSED) && !mContext.isFinishing) {

                    val bundle = Bundle()
                    bundle.let {
                        it.putString("dialog_first_text_line", alertTitleText)
                        it.putString("dialog_header_text_line", alertBodyText)
                        it.putString("survey_hash", collectorHash)
                        it.putString("custom_variables", if(customVariables.isNotEmpty())  customVariables.toString() else "")
                        it.putInt("request_code", requestCode)
                        it.putLong("after_decline_interval", afterDeclineInterval)
                        it.putLong("after_accept_interval", afterAcceptInterval)
                    }
                    val dialogFragment = SurveyDialogFragment()
                    dialogFragment.clickListener = this@SurveyMonkey
                    dialogFragment.apply {
                        arguments = bundle
                        show((mContext as FragmentActivity).supportFragmentManager,"")
                    }
                }
            }
        } catch (e: JSONException) {
            e.printStackTrace()
        }
    }

    fun startSMFeedbackActivityForResult(context: Activity, requestCode: Int, collectorHash: String, vararg customVariables: JSONObject?){
        runBlocking {
            return@runBlocking surveyManager.startSMFeedbackActivityForResult(context, requestCode, collectorHash, *customVariables)

        }
    }


    /**
     * Use the newSMFeedbackFragmentInstance method to create an SMFeedbackFragment which can be embedded in activity and used to present a survey to the user
     *
     * @param collectorHash The collector hash that points to your Mobile SDK Collector - found on the Collector Details page of your chosen collector on SurveyMonkey.com
     *
     * Once the user completes your survey, a JSONObject containing the user's response will be returned to the spawning activity in onActivityResult
     *
     * **Note:** If there are problems collecting the user's response, the SDK will return an SMError object with one of the domains and error codes listed in the SMError documentation
     * @param customVariables (Optional) PLATINUM USERS ONLY: If you've set up custom variables in Create, you can pass a JSON object with the format {"key":"value",...} here.
     * @see SMError
     */
    fun newSMFeedbackFragmentInstance(collectorHash: String?, vararg customVariables: JSONObject?): SMFeedbackFragment? {
        val customVariablesObj = if (customVariables.isNotEmpty()) customVariables[0] else null
        return SMFeedbackFragment.newInstance(
            buildURL(collectorHash!!, customVariablesObj),
            null,
            false
        )
    }

    override suspend fun onPositiveButtonClick(requestCode: Int, collectorHash: String, vararg customVariables: JSONObject?) {
        startSMFeedbackActivityForResult(mContext, requestCode, collectorHash, *customVariables)
    }

    override fun onNegativeButtonClick() {
        TODO("Not yet implemented")
    }

}