/*
 * $Id: TeeFilter.java,v 1.4 2007/10/09 11:45:56 kumarjayanti Exp $
 */

/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://glassfish.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems Inc. All Rights Reserved
 */
package com.sun.xml.wss.impl.filter;

import com.sun.xml.wss.logging.LogDomainConstants;
import java.io.ByteArrayInputStream;
import java.io.OutputStream;
import java.util.logging.Level;

import javax.xml.soap.SOAPMessage;

import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import com.sun.xml.wss.XWSSecurityException;
import java.util.logging.Logger;

/**
 * Copies the SOAP message into an OutputStream using an optional stylesheet
 * to format the message.  The original message is not modified.  This is
 * analogous to the "tee" unix command.
 *
 * @author Edwin Goei
 */
public class TeeFilter {
    // TODO Fix the stylesheet to pretty print a SOAP Message
    private static Logger log = Logger.getLogger(
            LogDomainConstants.WSS_API_DOMAIN,
            LogDomainConstants.WSS_API_DOMAIN_BUNDLE);
    private static final String prettyPrintStylesheet =
    "<xsl:stylesheet xmlns:xsl='http://www.w3.org/1999/XSL/Transform' xmlns:wsse='http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd'\n"
    + "  version='1.0'>\n"
    + "  <xsl:output method='xml' indent='yes'/>\n"
    + "  <xsl:strip-space elements='*'/>\n"
    + "  <xsl:template match='/'>\n"
    + "    <xsl:apply-templates/>\n"
    + "  </xsl:template>\n"
    + "  <xsl:template match='node() | @*'>\n"
    + "    <xsl:choose>\n"
    + "      <xsl:when test='contains(name(current()), \"wsse:Password\")'>\n"
    + "        <wsse:Password Type='{@Type}'>****</wsse:Password>\n"
    + "      </xsl:when>\n"
    + "      <xsl:otherwise>\n"
    + "        <xsl:copy>\n"
    + "          <xsl:apply-templates select='node() | @*'/>\n"
    + "        </xsl:copy>\n"
    + "      </xsl:otherwise>\n"
    + "    </xsl:choose>\n"
    + "  </xsl:template>\n"
    + "</xsl:stylesheet>\n";
    
    /** OutputStream for output. */
    private OutputStream out;
    
    /** Represents a stylesheet */
    private Templates templates;
    
    
    /**
     * Copy and optionally format a message
     *
     * @param out destination OutputStream
     * @param stylesheet XSLT stylesheet for format or if null, then does
     *     not format
     */
    public TeeFilter(OutputStream out, Source stylesheet)
    throws XWSSecurityException {
        init(out, stylesheet);
    }
    
    /**
     * Copy and optionally pretty print a message
     *
     * @param out destination OutputStream
     * @param prettyPrint true means to use built-in pretty print stylesheet
     * @throws XWSSecurityException
     */
    public TeeFilter(OutputStream out, boolean prettyPrint)
    throws XWSSecurityException {
        if (prettyPrint) {
            init(out, getPrettyPrintStylesheet());
        } else {
            init(out, null);
        }
    }
    
    /**
     * Saves a copy of message to Outputstream out
     *
     * @param out
     * @throws XWSSecurityException
     */
    public TeeFilter(OutputStream out) throws XWSSecurityException {
        init(out, null);
    }
    
    /**
     * A no-op
     *
     * @throws XWSSecurityException
     */
    public TeeFilter() throws XWSSecurityException {
        init(null, null);
    }
    
    private void init(OutputStream out, Source stylesheet)
    throws XWSSecurityException {
        this.out = out;
        
        if (stylesheet == null) {
            templates = null;
        } else {
            TransformerFactory tf = TransformerFactory.newInstance();
            //new com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl();
            try {
                templates = tf.newTemplates(stylesheet);
            } catch (TransformerConfigurationException e) {
                log.log(Level.SEVERE, "WSS0147.unableto.use.stylesheet",
                new Object[] {e.getMessage()});
                throw new XWSSecurityException("Unable to use stylesheet", e);
            }
        }
    }
    
    private Source getPrettyPrintStylesheet() {
        //        if (true) {
        //            if (defaultStylesheetSource == null) {
        //                byte[] xsltBytes = defaultStylesheet.getBytes();
        //                ByteArrayInputStream bais = new ByteArrayInputStream(xsltBytes);
        //                defaultStylesheetSource = new StreamSource(bais);
        //            }
        //            return defaultStylesheetSource;
        //        } else {
        byte[] xsltBytes = prettyPrintStylesheet.getBytes();
        ByteArrayInputStream bais = new ByteArrayInputStream(xsltBytes);
        Source stylesheetSource = new StreamSource(bais);
        return stylesheetSource;
        //        }
    }
    
    /**
     * Invokes the MessageFilter on the SOAPMessage sm.  A
     * XWSSecurityException is thrown if the operation did not succeed.
     *
     * @param secureMessage SOAPMessage to perform the operation on
     *
     * @throws com.sun.xml.wss.XWSSecurityException if the operation did not
     *    succeed
     */
    public void process(SOAPMessage secureMessage) throws XWSSecurityException {
        if (out == null) {
            return;
        }
        
        Transformer transformer;
        try {
            if (secureMessage.countAttachments() > 0) {
                secureMessage.writeTo(out);
            } else {
                if (templates == null) {
                    // Use identity transform
                    transformer =
                    new com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl()
                    .newTransformer();
                } else {
                    // Use supplied stylesheet via Templates object
                    transformer = templates.newTransformer();
                }
                Source msgSource = secureMessage.getSOAPPart().getContent();
                transformer.transform(msgSource, new StreamResult(out));
            }
        } catch (Exception ex) {
            log.log(Level.SEVERE, "WSS0148.unableto.process.soapmessage",
            new Object[] {ex.getMessage()});
            throw new XWSSecurityException("Unable to process SOAPMessage", ex);
        }
    }
}
