/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2007 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.grizzly.util;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;

/**
 * Class contains set of useful operations commonly used in the framework
 * 
 * @author Alexey Stashok
 * @author Jean-Francois Arcand
 */
public class Utils {
    /**
     * Method reads data from <code>SelectableChannel</code> to 
     * <code>ByteBuffer</code>. If data is not immediately available - channel
     *  will be reregistered on temporary <code>Selector</code> and wait maximum
     * readTimeout milliseconds for data.
     * 
     * @param channel <code>SelectableChannel</code> to read data from
     * @param byteBuffer <code>ByteBuffer</code> to store read data to
     * @param readTimeout maximum time in millis operation will wait for 
     * incoming data
     * 
     * @return number of bytes were read
     * @throws <code>IOException</code> if any error was occured during read
     */
    public static int readWithTemporarySelector(SelectableChannel channel,
            ByteBuffer byteBuffer, long readTimeout) throws IOException {
        int count = 1;
        int byteRead = 0;
        int preReadInputBBPos = byteBuffer.position();
        Selector readSelector = null;
        SelectionKey tmpKey = null;

        try {
            ReadableByteChannel readableChannel = (ReadableByteChannel) channel;
            while (count > 0){
                count = readableChannel.read(byteBuffer);
                if ( count > -1 )
                    byteRead += count;
                else
                    byteRead = count;
            }            
            
            if (byteRead == 0 && byteBuffer.position() == preReadInputBBPos) {
                readSelector = SelectorFactory.getSelector();

                if ( readSelector == null ){
                    return 0;
                }
                count = 1;
                
                tmpKey = channel.register(readSelector, SelectionKey.OP_READ);
                tmpKey.interestOps(tmpKey.interestOps() | SelectionKey.OP_READ);
                int code = readSelector.select(readTimeout);
                tmpKey.interestOps(
                    tmpKey.interestOps() & (~SelectionKey.OP_READ));

                if ( code == 0 ){
                    return 0; // Return on the main Selector and try again.
                }

                while (count > 0){
                    count = readableChannel.read(byteBuffer);
                    if ( count > -1 )
                        byteRead += count;
                    else
                        byteRead = count;                    
                }
            } else if (byteRead == 0 && byteBuffer.position() != preReadInputBBPos) {
                byteRead += (byteBuffer.position() - preReadInputBBPos);
            }
        } finally {
            if (tmpKey != null)
                tmpKey.cancel();

            if ( readSelector != null) {
                // Bug 6403933
                try{
                    readSelector.selectNow();
                } catch (IOException e) {
                }
                SelectorFactory.returnSelector(readSelector);
            }
        }

        return byteRead;
    } 
}
