/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2007 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.grizzly;

import java.io.IOException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

/**
 * ReadController class represents <code>Controller</code>,
 * which is not itself independent.
 *
 * Should be used for handling OP_READ operations
 * Supports TCP derived protocols
 *
 * @author Alexey Stashok
 */
public class ReadController extends Controller {

    /**
     * Add a <code>Channel</code>
     * to be processed by <code>ReadController</code>'s
     * <code>SelectorHandler</code>
     *
     * @param channel new channel to be managed by ReadController
     * @param protocol name of the protocol channel corresponds to
     */
    public void addChannel(SelectableChannel channel, Protocol protocol) {
        SelectorHandler selectorHandler = getSelectorHandler(protocol);
        selectorHandler.register(channel, SelectionKey.OP_READ);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void doSelect(SelectorHandler selectorHandler) {
        // Create <code>Selector</code> in the same Thread it is processed
        if(selectorHandler.getSelector() == null) {
            try {
                selectorHandler.setSelector(Selector.open());
            } catch(IOException e) {
                notifyException(e);
                logger().log(Level.SEVERE, "doSelect exception", e);
            }
        }
        
        super.doSelect(selectorHandler);
    }
    
    /**
     * Start the Controller. If the Pipeline and/or Handler has not been
     * defined, the default will be used.
     */
    @Override
    public void start() throws IOException {
        stateLock.lock();
        try {
            // could be null if ReadController will be used outside main Controller
            if (state == null) {
                state = State.STARTED;
            }
        } finally {
            stateLock.unlock();
        }
        notifyStarted();

        int selectorHandlerCount = selectorHandlers.size();
        readySelectorHandlerCounter = new AtomicInteger(selectorHandlerCount);
        stopedSelectorHandlerCounter = new AtomicInteger(selectorHandlerCount);

        for (SelectorHandler selectorHandler : selectorHandlers) {
            Runnable selectorRunner = new SelectorHandlerRunner(this, selectorHandler);
            if (selectorHandlerCount > 1) {
                // if there are more than 1 selector handler - run it in separate thread
                new Thread(selectorRunner).start();
            } else {
                // else run it in current thread
                selectorRunner.run();
            }
        }

        waitUntilSeletorHandlersStop();
        selectorHandlers.clear();
    }
}