package com.sudicode.githubextensions;

import com.google.gson.JsonParser;
import com.sudicode.githubextensions.global.Environment;
import com.sudicode.githubextensions.service.PayloadHandler;
import org.apache.commons.codec.digest.HmacUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.security.MessageDigest;

/**
 * Handles webhook requests. In GitHub, you must set the payload URL to <code>(Server URL...)/github-webhook</code>.
 */
@RestController
@RequestMapping("/github-webhook")
public class WebhookController {

    private static final Logger LOG = LoggerFactory.getLogger(WebhookController.class);
    private static final int SIGNATURE_LENGTH = 45;

    private final PayloadHandler payloadHandler;
    private final JsonParser jsonParser;

    @Autowired
    public WebhookController(PayloadHandler payloadHandler,
                             JsonParser jsonParser) {
        this.payloadHandler = payloadHandler;
        this.jsonParser = jsonParser;
    }

    /**
     * Handle a POST request. The request <strong>must</strong> originate from GitHub; this will be verified via the
     * "X-Hub-Signature" header.
     *
     * @param signature "X-Hub-Signature" header
     * @param payload   Request payload, formatted as a String
     * @return A {@link ResponseEntity}
     */
    @RequestMapping(method = RequestMethod.POST)
    public ResponseEntity<HttpStatus> postRequest(@RequestHeader("X-Hub-Signature") String signature, @RequestBody String payload) {
        // Validate payloads from GitHub
        if (signature == null) {
            return new ResponseEntity<>(HttpStatus.BAD_REQUEST); // no signature
        }
        String computed = String.format("sha1=%s", HmacUtils.hmacSha1Hex(Environment.SECRET_TOKEN, payload));
        boolean invalidLength = signature.length() != SIGNATURE_LENGTH;
        if (invalidLength || !MessageDigest.isEqual(signature.getBytes(), computed.getBytes())) {
            return new ResponseEntity<>(HttpStatus.UNAUTHORIZED); // incorrect signature
        }

        // Handle payload
        try {
            payloadHandler.handle(jsonParser.parse(payload).getAsJsonObject());
        } catch (Throwable e) {
            LOG.error(e.getMessage(), e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
        return new ResponseEntity<>(HttpStatus.NO_CONTENT);
    }

}
