/*
 * Decompiled with CFR 0.152.
 */
package com.slack.api.methods.impl;

import com.slack.api.SlackConfig;
import com.slack.api.methods.MethodsCompletionException;
import com.slack.api.methods.MethodsConfig;
import com.slack.api.methods.SlackApiException;
import com.slack.api.methods.SlackApiResponse;
import com.slack.api.methods.impl.AsyncExecutionSupplier;
import com.slack.api.methods.impl.AsyncRateLimitQueue;
import com.slack.api.methods.impl.TeamIdCache;
import com.slack.api.methods.impl.ThreadPools;
import com.slack.api.methods.metrics.MetricsDatastore;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncRateLimitExecutor {
    private static final Logger log = LoggerFactory.getLogger(AsyncRateLimitExecutor.class);
    private static final ConcurrentMap<String, AsyncRateLimitExecutor> ALL_EXECUTORS = new ConcurrentHashMap<String, AsyncRateLimitExecutor>();
    private final MethodsConfig config;
    private final MetricsDatastore metricsDatastore;
    private final TeamIdCache teamIdCache;
    private static final List<String> NO_TOKEN_METHOD_NAMES = Arrays.asList("api.test", "oauth.access", "oauth.token", "oauth.v2.access");

    private AsyncRateLimitExecutor(SlackConfig config) {
        this.config = config.getMethodsConfig();
        this.metricsDatastore = config.getMethodsConfig().getMetricsDatastore();
        this.teamIdCache = new TeamIdCache(config);
    }

    public static AsyncRateLimitExecutor get(String executorName) {
        return (AsyncRateLimitExecutor)ALL_EXECUTORS.get(executorName);
    }

    public static AsyncRateLimitExecutor getOrCreate(SlackConfig config) {
        AsyncRateLimitExecutor executor = (AsyncRateLimitExecutor)ALL_EXECUTORS.get(config.getMethodsConfig().getExecutorName());
        if (executor == null) {
            executor = new AsyncRateLimitExecutor(config);
            ALL_EXECUTORS.put(config.getMethodsConfig().getExecutorName(), executor);
        }
        return executor;
    }

    public <T extends SlackApiResponse> CompletableFuture<T> execute(String methodName, Map<String, String> params, AsyncExecutionSupplier<T> methodsSupplier) {
        String token = params.get("token");
        String teamId = token != null ? this.teamIdCache.lookupOrResolve(token) : null;
        ExecutorService executorService = teamId != null ? ThreadPools.getOrCreate(this.config, teamId) : ThreadPools.getDefault(this.config);
        return CompletableFuture.supplyAsync(() -> {
            if (NO_TOKEN_METHOD_NAMES.contains(methodName)) {
                return this.runWithoutQueue(teamId, methodName, methodsSupplier);
            }
            String messageId = AsyncRateLimitExecutor.generateMessageId();
            String methodNameWithSuffix = this.toMethodNameWithSuffix(methodName, params);
            this.addMessageId(teamId, methodNameWithSuffix, messageId);
            this.initCurrentQueueSizeStatsIfAbsent(teamId, methodNameWithSuffix);
            return this.enqueueThenRun(messageId, teamId, methodName, params, methodsSupplier);
        }, executorService);
    }

    private void initCurrentQueueSizeStatsIfAbsent(String teamId, String methodNameWithSuffix) {
        if (teamId != null) {
            this.metricsDatastore.setCurrentQueueSize(this.config.getExecutorName(), teamId, methodNameWithSuffix, 0);
        }
    }

    private void addMessageId(String teamId, String methodNameWithSuffix, String messageId) {
        this.metricsDatastore.addToWaitingMessageIds(this.config.getExecutorName(), teamId, methodNameWithSuffix, messageId);
    }

    private void removeMessageId(String teamId, String methodNameWithSuffix, String messageId) {
        this.metricsDatastore.deleteFromWaitingMessageIds(this.config.getExecutorName(), teamId, methodNameWithSuffix, messageId);
    }

    private String toMethodNameWithSuffix(String methodName, Map<String, String> params) {
        String methodNameWithSuffix = methodName;
        if (methodName.equals("chat.postMessage")) {
            methodNameWithSuffix = "chat.postMessage_" + params.get("channel");
        }
        return methodNameWithSuffix;
    }

    private <T extends SlackApiResponse> T runWithoutQueue(String teamId, String methodName, AsyncExecutionSupplier<T> methodsSupplier) {
        try {
            return methodsSupplier.execute();
        }
        catch (RuntimeException e) {
            return AsyncRateLimitExecutor.handleRuntimeException(teamId, methodName, e);
        }
        catch (IOException e) {
            return AsyncRateLimitExecutor.handleIOException(teamId, methodName, e);
        }
        catch (SlackApiException e) {
            AsyncRateLimitExecutor.logSlackApiException(teamId, methodName, e);
            throw new MethodsCompletionException(null, e, null);
        }
    }

    private <T extends SlackApiResponse> T enqueueThenRun(String messageId, String teamId, String methodName, Map<String, String> params, AsyncExecutionSupplier<T> methodsSupplier) {
        try {
            AsyncRateLimitQueue activeQueue = AsyncRateLimitQueue.getOrCreate(this.config, teamId);
            if (activeQueue == null) {
                log.warn("Queue for teamId: {} was not found. Going to run the API call immediately.", (Object)teamId);
            }
            AsyncExecutionSupplier supplier = null;
            activeQueue.enqueue(messageId, teamId, methodName, params, methodsSupplier);
            for (long consumedMillis = 0L; supplier == null && consumedMillis < (long)this.config.getMaxIdleMills(); consumedMillis += 10L) {
                Thread.sleep(10L);
                supplier = activeQueue.dequeueIfReady(messageId, teamId, methodName, params);
                this.removeMessageId(teamId, this.toMethodNameWithSuffix(methodName, params), messageId);
            }
            if (supplier == null) {
                activeQueue.remove(methodName, messageId);
                throw new RejectedExecutionException("Gave up executing the message after " + this.config.getMaxIdleMills() + " milliseconds.");
            }
            Object response = supplier.execute();
            return response;
        }
        catch (RuntimeException e) {
            return AsyncRateLimitExecutor.handleRuntimeException(teamId, methodName, e);
        }
        catch (IOException e) {
            return AsyncRateLimitExecutor.handleIOException(teamId, methodName, e);
        }
        catch (SlackApiException e) {
            AsyncRateLimitExecutor.logSlackApiException(teamId, methodName, e);
            if (e.getResponse().code() == 429) {
                return this.enqueueThenRun(messageId, teamId, methodName, params, methodsSupplier);
            }
            throw new MethodsCompletionException(null, e, null);
        }
        catch (InterruptedException e) {
            log.error("Got an InterruptedException (error: {})", (Object)e.getMessage(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private static String generateMessageId() {
        return UUID.randomUUID().toString();
    }

    private static <T extends SlackApiResponse> T handleRuntimeException(String teamId, String methodName, RuntimeException e) {
        log.error("Got an exception while calling {} API (team: {}, error: {})", new Object[]{methodName, teamId, e.getMessage(), e});
        throw new MethodsCompletionException(null, null, e);
    }

    private static <T extends SlackApiResponse> T handleIOException(String teamId, String methodName, IOException e) {
        log.error("Failed to connect to {} API (team: {}, error: {})", new Object[]{methodName, teamId, e.getMessage(), e});
        throw new MethodsCompletionException(e, null, null);
    }

    private static void logSlackApiException(String teamId, String methodName, SlackApiException e) {
        if (e.getResponse().code() == 429) {
            String retryAfterSeconds = e.getResponse().header("Retry-After");
            log.error("Got a rate-limited response from {} API (team: {}, error: {}, retry-after: {})", new Object[]{methodName, teamId, e.getMessage(), retryAfterSeconds, e});
        } else {
            log.error("Got an unsuccessful response from {} API (team: {}, error: {}, status code: {})", new Object[]{methodName, teamId, e.getMessage(), e.getResponse().code(), e});
        }
    }
}

