/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 03.07.2009 - [RH] - creation
 */
package javax.rad.model.event;

import javax.rad.model.IDataBook;
import javax.rad.model.IDataRow;

/**
 * The <code>DataBookEvent</code> gives information about changes in the 
 * {@link IDataBook}.
 * 
 * @see javax.rad.model.IDataBook
 * 
 * @author Martin Handsteiner
 */
public class DataBookEvent
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** Specifies the type of change. */
	public enum ChangedType 
	{
		/** selected row will be change. */
		BEFORE_ROW_SELECTED,
		/** selected row has been changed. */
		AFTER_ROW_SELECTED,
		
		/** <code>IDataRow</code> will be inserted in memory. */
		BEFORE_INSERTING,
		/** <code>IDataRow</code> has been inserted in memory. */
		AFTER_INSERTING,
		/** <code>IDataRow</code> will be inserted into the storage. */
		BEFORE_INSERTED,
		/** <code>IDataRow</code> has been inserted into the storage. */
		AFTER_INSERTED,

		/** <code>IDataRow</code> will be updated in memory. */
		BEFORE_UPDATING,
		/** <code>IDataRow</code> has been updated in memory. */
		AFTER_UPDATING,
		/** <code>IDataRow</code> will be updated into the storage. */
		BEFORE_UPDATED,
		/** <code>IDataRow</code> has been updated into the storage. */
		AFTER_UPDATED,
		
		/** <code>IDataRow</code> will be deleted in memory. */
		BEFORE_DELETING,
		/** <code>IDataRow</code> has been deleted in memory. */
		AFTER_DELETING,
		/** <code>IDataRow</code> will be deleted into the storage. */
		BEFORE_DELETED,
		/** <code>IDataRow</code> has been deleted into the storage. */
		AFTER_DELETED,

		/** <code>IDataRow</code> will be restored. */
		BEFORE_RESTORE,
		/** <code>IDataRow</code> has been restored. */
		AFTER_RESTORE,

		/** <code>IDataRow</code> will be refreshed. */
		BEFORE_RELOAD,
		/** <code>IDataRow</code> has been refreshed. */
		AFTER_RELOAD,

		/** selected column will be change. */
		BEFORE_COLUMN_SELECTED,
		/** selected column has been changed. */
		AFTER_COLUMN_SELECTED
	}
	
	/** The changed IDataBook. */
	private IDataBook changedDataBook;
	
	/** The changed type. */
	private ChangedType changedType;
	
	/** The original IDataRow. */
	private IDataRow originalDataRow;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>DataBookEvent</code>.
	 * 
	 * @param pChangedDataBook the changed IDataBook.
	 * @param pChangedType the type of change.
	 * @param pOriginalDataRow the old IDataRow.
	 */
	public DataBookEvent(IDataBook pChangedDataBook, ChangedType pChangedType, IDataRow pOriginalDataRow)
	{
		changedDataBook = pChangedDataBook;
		changedType = pChangedType;
		originalDataRow = pOriginalDataRow;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the {@link IDataBook} that is changed.
	 * 
	 * @return the {@link IDataBook} that is changed.
	 */
	public IDataBook getChangedDataBook()
	{
		return changedDataBook;
	}
	
	/**
	 * Gets the type of change.
	 * 
	 * @return the type of change.
	 */
	public ChangedType getChangedType()
	{
		return changedType;
	}
	
	/**
	 * Gets the original IDataRow before change.
	 * 
	 * @return the original IDataRow before change.
	 */
	public IDataRow getOriginalDataRow()
	{
		return originalDataRow;
	}
	
}	// DataBookEvent
