/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 11.11.2008 - [RH] - DeleteCascade added, notifyMasterReferenceColumnsChanged(IDataRow) add
 * 13.11.2008 - [RH] - is/setUpdate/Delete/InsertEnabled() added; is/setReadOnly added 
 * 17.11.2008 - [RH] - Selection mode added; SetDefaultDelected removed
 *                     refresh removed(); store renamed to Save...(); other methods renamed
 * 10.04.2009 - [RH] - interface review - derived now from IChangeableDataRow
 * 16.04.2009 - [RH] - remove/add/getDetailDataBooks moved to IDataBook.    
 * 19.04.2009 - [RH] - Enum SelectionMode and WriteBackIsolationLevel          
 * 06.09.2010 - [RH] - comment corrected for getDataPage(...)   
 * 15.10.2010 - [RH] - detailChanged renamed to notifyDetailChanged 
 * 29.11.2010 - [JR] - searchNext added
 *                   - isInsertEnabled, isUpdateEnabled, isDeleteEnabled throws ModelException 
 * 13.05.2011 - [RH] - #350 - MemDataBook should remove all details in saveSelectedRow(), if a row is deleted   
 * 08.04.2013 - [RH] - saveDataPage() add (during saveAllDataRows Bug fix)    
 * 12.04.2013 - [RH] - Need isUpdate/Delete/InsertAllowed and isUpdate/Delete/InsertEnabled should only get what is set - realized!  
 */
package javax.rad.model;

import javax.rad.model.condition.ICondition;
import javax.rad.model.event.DataBookHandler;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.util.INamedObject;

/**
 * The <code>IDataBook</code> is a storage independent table, and handles all operations
 * to load, save and manipulate table oriented data. <br>
 * An <code>IDataBook</code> has at least one <code>IDataPage</code> to store all 
 * <code>IDataRow</code>'s of this <code>IDataBook</code>. If an <code>IDataBook</code> 
 * has detail <code>IDataBook</code>'s, it handles for each detail <code>IDataBook</code> 
 * the <code>IDataPage</code>'s. Thats necessary because every change of the selected
 * <code>IDataRow</code> needs to initiate the change of the corresponding details. 
 * So the <code>IDataBook</code> adds the appropriate <code>IDataPage</code>'s into the 
 * detail <code>IDataBook</code>'s.<br>
 * The <code>IDataBook</code> is also a <code>IChangeableDataRow</code>, the selected row 
 * of the <code>IDataBook</code>.
 * 
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IRowDefinition
 * @see javax.rad.model.IChangeableDataRow
 * @see javax.rad.model.IDataSource
 * 
 * @author Roland Hrmann
 */
public interface IDataBook extends IChangeableDataRow, 
                                   IDataPage,
                                   INamedObject
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * The IDataBook selection modes specify which row should be selected after reload or
	 * after the master row has changed. 
	 */
	public enum SelectionMode
	{
		/**
		 * means, that after reload, set sort, set filter or change of the master row, that no row (-1) will 
		 * be selected. 
		 */
		DESELECTED, 
		/** 
		 * means, that after reload, set sort, set filter or change of the master row, that the first row (0) will 
		 * be selected. 
		 */
		FIRST_ROW, 
		/** 
		 * means, that after reload or set sort the current row 
		 * (last selected row) or if not possible, on set filter or change of master row the first row (0) will be selected. 
		 */
		CURRENT_ROW,
		/** 
		 * means, that after reload or set sort the current row 
		 * (last selected row) or if not possible, on set filter or change of master row no row (-1) will be selected. 
		 */
		CURRENT_ROW_DESELECTED,
		/** 
		 * means, that after reload, set sort or on set filter the current row 
		 * (last selected row) or if not possible or change of master row the first row (0) will be selected. 
		 * Be aware that current_row mode on set filter can cause performance issues, as all rows have to be fetched, if the row can not be found.
		 */
		CURRENT_ROW_SETFILTER,
		/** 
		 * means, that after reload, set sort or on set filter the current row 
		 * (last selected row) or if not possible or change of master row no row (-1) will be selected. 
		 * Be aware that current_row mode on set filter can cause performance issues, as all rows have to be fetched, if the row can not be found.
		 */
		CURRENT_ROW_DESELECTED_SETFILTER
	}
	
	/** 
	 * The IDataBook (and IDataSourrce as Default) write back isolations levels, specify 
	 * when the changes in the IDataBook should be implicit write back to the storage. 
	 * A typical way and default is data row level (DATA_ROW).
	 */
	public enum WriteBackIsolationLevel
	{
		/** 
		 * The <code>IDataSource</code> stores the data when another <code>IDataRow</code> 
		 * has changed/selected. 
		 */
		DATA_ROW,
		/** 
		 * The <code>IDataSource</code> stores the data when a <code>saveXXXX()</code> 
		 * method is called. 
		 */
		DATASOURCE
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Method definitions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Sets the <code>IRowDefinition</code> of the <code>IDataBook</code>.
	 * 
	 * @param pRowDefinition
	 *            the new <code>IRowDefinition</code>
	 * @throws ModelException
	 * 			  if the <code>IRowDefinition</code> is empty 
	 */
	public void setRowDefinition(IRowDefinition pRowDefinition) throws ModelException;

	/**
	 * Sets the <code>IDataSource</code> for this <code>IDataBook</code>.<br>
	 * The <code>IDataSource</code> defines the access to the storage like DB, XML, ...
	 * 
	 * @see javax.rad.model.IDataSource
	 * @param pDataSource
	 *            the <code>IDataSource</code> to use for load/save data	 	 
 	 * @throws ModelException
	 *             if the <code>IDataBook</code> is already opened. -> Not allowed
     */
	public void setDataSource(IDataSource pDataSource) throws ModelException;

	/**
	 * Returns the <code>IDataSource</code> for this <code>IDataBook</code>.
	 * 
	 * @return the <code>IDataSource</code> for this <code>IDataBook</code>.
	 */
	public IDataSource getDataSource();
	
	/**
	 * Set the name of the <code>IDataBook</code>.
	 * 
	 * @param pName
	 *            <code>IDataBook</code> name
 	 * @throws ModelException
	 *             if the <code>IDataBook</code> is already opened. -> Not allowed
	 */
	public void setName(String pName) throws ModelException;

	/**
	 * Returns the <code>IDataBook</code> name.
	 * 
	 * @return the <code>IDataBook</code> name.
	 */
	public String getName();
	
	/**
	 * Sets when the <code>IDataSource</code> needs to write back the 
	 * data to the storage.<br>
	 * The default write back isolation level is <code>DATA_ROW</code>.
	 * 
	 * @see javax.rad.model.IDataBook.WriteBackIsolationLevel
	 * @param pIsolationLevel
	 *            the new write back isolation level.
	 */
	public void setWritebackIsolationLevel(WriteBackIsolationLevel pIsolationLevel);

	/**
	 * Returns the current write back isolation level.
	 * 
	 * @return the current write back isolation level.
	 */	
	public WriteBackIsolationLevel getWritebackIsolationLevel();
	
	/**
	 * Sets the master <code>IDataBook</code> to this detail <code>IDataBook</code>
	 * over a <code>ReferenceDefinition</code>.
	 * 
	 * @see javax.rad.model.reference.ReferenceDefinition
	 * @param pReferenceDefinitionToMasterBook
	 *            the <code>ReferenceDefinition</code> to the master <code>IDataBook</code>.
	 * @throws ModelException 
	 * 			  if the <code>IDataBook</code> is open
	 */
	public void setMasterReference(ReferenceDefinition pReferenceDefinitionToMasterBook) throws ModelException;

	/**
	 * Returns the <code>ReferenceDefinition</code> to the master <code>IDataBook</code>.
	 * 
	 * @return the <code>ReferenceDefinition</code> to the master <code>IDataBook</code>.
	 */
	public ReferenceDefinition getMasterReference();

	/**
	 * True, if this IDataBook is self joined.
	 * 
	 * @return True, if this IDataBook is self joined.
	 */
	public boolean isSelfJoined();
	
	/**
	 * Gets the IDataPage for the given TreePath.
	 * 
	 * @param pTreePath the TreePath.
	 * @return gets the tree root reference.
	 * @throws ModelException 
	 * 			  if the pRootRow or pTreePath don't contains the master columns from the master
	 *            <code>ReferenceDefinition</code> 
	 */
	public IDataPage getDataPage(TreePath pTreePath) throws ModelException;

	/**
	 * Gets the IDataPage for the given TreePath and root row.
	 * 
	 * @param pRootRow the root row.
	 * @param pTreePath the TreePath.
	 * @return gets the tree root reference.
	 * @throws ModelException 
	 * 			  if the pRootRow or pTreePath don't contains the master columns from the master
	 *            <code>ReferenceDefinition</code> 
	 */
	public IDataPage getDataPage(IDataRow pRootRow, TreePath pTreePath) throws ModelException;

	/**
	 * Returns the corresponding <code>IDataPage</code> to specified root row
	 * from the root <code>DataBook</code>. If it doesn't exists, it will be created
	 * and returned.
	 * 
	 * @param pRootRow
	 * 				the root <code>IDataRow</code> of the root <code>DataBook</code>.
	 * @throws ModelException 
	 * 			  if the pRootRow don't contains the master columns from the master
	 *            <code>ReferenceDefinition</code> 
	 * @return the corresponding <code>IDataPage</code> to specified master row
	 *         from the master <code>DataBook</code>.
	 */
	public IDataPage getDataPageWithRootRow(IDataRow pRootRow) throws ModelException;

	/**
	 * Gets the tree root reference.
	 * 
	 * @return gets the tree root reference.
	 */
	public ReferenceDefinition getRootReference();
	
	/**
	 * Sets the tree root reference.
	 * 
	 * @param pReferenceDefinition the tree root reference.
	 * @throws ModelException 
	 * 			  if the <code>IDataBook</code> is open
	 */
	public void setRootReference(ReferenceDefinition pReferenceDefinition) throws ModelException;
	
	/**
	 * Gets the tree current tree path.
	 * 
	 * @return gets the tree root refernce.
	 */
	public TreePath getTreePath();
	
	/**
	 * Sets the tree current tree path.
	 * 
	 * @param pTreePath the tree root reference.
	 * @throws ModelException 
	 * 			  if the <code>IDataBook</code> is open
	 */
	public void setTreePath(TreePath pTreePath) throws ModelException;

	/**
	 * Returns the corresponding <code>IDataPage</code> to specified master row
	 * from the master <code>DataBook</code>. If it doesn't exists, it will be created
	 * and returned.
	 * 
	 * @param pMasterRow
	 * 				the master <code>IDataRow</code> of the master <code>DataBook</code>.
	 * @throws ModelException 
	 * 			  if the pMasterRow don't contains the master columns from the master
	 *            <code>ReferenceDefinition</code> 
	 * @return the corresponding <code>IDataPage</code> to specified master row
	 *         from the master <code>DataBook</code>.
	 */
	public IDataPage getDataPage(IDataRow pMasterRow) throws ModelException;
	
	/**
	 * Returns true if an <code>IDataPage</code> to specified master row
	 * from the master <code>DataBook</code> exists.
	 * 
	 * @param pMasterDataRow
	 * 				the master <code>IDataRow</code> of the master <code>DataBook</code>.
	 * @throws ModelException 
	 * 			  if the pMasterRow don't contains the master columns from the master
	 *            <code>ReferenceDefinition</code> 
	 * @return true if an <code>IDataPage</code> to specified master row
	 * 		   from the master <code>DataBook</code> exists.
	 */
	public boolean hasDataPage(IDataRow pMasterDataRow) throws ModelException;
	
	/**
	 * Returns the selection mode for this IDataBook.
	 * 
	 * @return the selection mode for this IDataBook.
	 */
	public SelectionMode getSelectionMode();
	
	/**
	 * Sets the one of the following selection mode for this IDataBook.<br>
	 * DESELECTED   means, that after reload or change of the master row, that no row (-1) will be selected.<br>
	 * FIRST_ROW 	means, that after reload or change of the master row, that the first row (0) will be selected.<br>
	 * CURRENT_ROW  means, that after reload or change of the master row, that the current row (last selected row)<br>
	 *              or if not possible the first row (0) will be selected.
	 * 
	 * @param pSelectionMode
	 * 				the selection mode for this IDataBook.
	 */
	public void setSelectionMode(SelectionMode pSelectionMode);	
	
	/**
	 * Sets the selected row index in the <code>IDataBook</code>.<br>
	 * If the row is not in memory, it is loaded from the storage.<br>
	 * <br>
	 * The write back isolation level needs to be considered. This means if the
	 * level is <code>WriteBackIsolationLevel.DATA_ROW</code>, the dataSource.storeAllDataBooks() method 
	 * needs to be called.<br>
	 * Implementations need to handle master-detail <code>IDataBook</code> behavior.
	 * <br><br>
	 * Events:<br>
	 * Before the selected row is changed, the IControl.saveEditing() event is called.<br>
	 * Before the selected row is changed, the IDataBookListener.BEFORE_ROW_SELECTED event is raised.<br>
	 * After the selected row is changed, the IDataBookListener.AFTER_ROW_SELECTED event is raised.<br>
	 * After the selected row is changed, all registered IControls get called the notifyRepaint() method.
	 * 
	 * @param pDataRowIndex
	 *            the selected row index in the <code>IDataBook</code>.
	 * @throws ModelException
	 *            if the row with the iDataRowIndex couldn't get from the
	 *            storage
	 *            or if the <code>IDataBook</code> isn't open or the master 
	 *            <code>IDataBook</code> has no selected row.
	 */
	public void setSelectedRow(int pDataRowIndex) throws ModelException;

	/**
	 * Returns the selected row index in the <code>IDataBook</code>.<br>
	 * This should be used from the GUI controls to show which one is selected.
	 * 
	 * @return the selected row index in the <code>IDataBook</code> or -1 if no row is selected.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
	 */
	public int getSelectedRow() throws ModelException;
	
	/**
	 * Sets the selected column in the <code>IDataBook</code>.<br>
	 * Events:<br>
	 * Before the selected column is changed, the IDataBookListener.BEFORE_COLUMN_SELECTED event is raised.<br>
	 * After the selected row is changed, the IDataBookListener.AFTER_COLUMN_SELECTED event is raised.<br>
	 * 
	 * @param pSelectedColumn
	 *            the selected column in the <code>IDataBook</code>.
	 * @throws ModelException
	 *            if the column does not exist.
	 *            or if the <code>IDataBook</code> isn't open or the master 
	 *            <code>IDataBook</code> has no selected row.
	 */
	public void setSelectedColumn(String pSelectedColumn) throws ModelException;

	/**
	 * Returns the selected column in the <code>IDataBook</code>.<br>
	 * This should be used from the GUI controls to show which one is selected.
	 * 
	 * @return the selected column in the <code>IDataBook</code>.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
	 */
	public String getSelectedColumn() throws ModelException;
	
	/**
	 * Sets if all rows of this detail table should be deleted, when the master row is deleted.<br>
	 * 
	 * @param pDeleteCascade
	 *            the value true means, delete the rows.
	 */
	public void setDeleteCascade(boolean pDeleteCascade);

	/**
	 * Returns true if all rows of this detail table should be deleted, when the master row is deleted.<br>
	 * 
	 * @return true if all rows of this detail table should be deleted, when the master row is deleted.
	 */
	public boolean isDeleteCascade();
	
	/**
	 * Returns true if insert is allowed on this IDataBook.
	 * Its the case if isInsertEnabled()==true and a insert is possible.
	 * 
	 * @return true if insert is allowed on this IDataBook.
	 * @throws ModelException if insert is not possible because row access failed
	 */
	public boolean isInsertAllowed() throws ModelException;
	
	/**
	 * Returns true if insert is enabled on this IDataBook.
	 * 
	 * @return true if insert is enabled on this IDataBook.
	 * @throws ModelException if insert is not possible because row access failed
	 */
	public boolean isInsertEnabled() throws ModelException;

	/**
	 * Sets whether insert is enabled on this IDataBook.
	 * 
	 * @param pInsertEnabled true if insert is allowed on this IDataBook.
	 */
	public void setInsertEnabled(boolean pInsertEnabled);

	/**
	 * Returns true if update is allowed on this IDataBook.
	 * Its the case if isUpdateEnabled()==true and a update is possible. (e.g. getSelectedRow() != -1)
	 * 
	 * @return true if update is allowed on this IDataBook.
	 * @throws ModelException if update is not possible because row access failed	 
	 */
	public boolean isUpdateAllowed() throws ModelException;

	/**
	 * Returns true if update is enabled on this IDataBook.
	 * 
	 * @return true if update is enabled on this IDataBook.
	 * @throws ModelException if update is not possible because row access failed	 
	 */
	public boolean isUpdateEnabled() throws ModelException;

	/**
	 * Sets whether update is enabled on this IDataBook.
	 * 
	 * @param pUpdateEnabled true if update is enabled on this IDataBook.
	 */
	public void setUpdateEnabled(boolean pUpdateEnabled);

	/**
	 * Returns true if delete is allowed on this IDataBook.
	 * Its the case if isDeleteEnabled()==true and a delete is possible. (e.g. getSelectedRow() != -1)
	 * 
	 * @return true if delete is allowed on this IDataBook.
	 * @throws ModelException if delete is not possible because row access failed
	 */
	public boolean isDeleteAllowed() throws ModelException;

	/**
	 * Returns true if delete is enabled on this IDataBook.
	 * 
	 * @return true if delete is enabled on this IDataBook.
	 * @throws ModelException if delete is not possible because row access failed
	 */
	public boolean isDeleteEnabled() throws ModelException;

	/**
	 * Sets whether delete is enabled on this IDataBook.
	 * 
	 * @param pDeleteEnabled true if delete is enabled on this IDataBook.
	 */
	public void setDeleteEnabled(boolean pDeleteEnabled);
	
	/**
	 * Returns true if this IDataBook is read only.
	 * 
	 * @return true if this IDataBook is read only.
	 */
	public boolean isReadOnly();

	/**
	 * Sets whether this IDataBook is read only.
	 * 
	 * @param pReadOnlyEnabled true if this IDataBook is read only.
	 * 
	 * @throws ModelException if the DataBook cannot be set read only.
	 */
	public void setReadOnly(boolean pReadOnlyEnabled) throws ModelException;	
	
	/**
	 * Returns true if the <code>IDataBook</code> is opened.
	 * 
	 * @return true if the <code>IDataBook</code> is opened.
	 */
	public boolean isOpen();
	
	/**
	 * Opens the <code>IDataBook</code>. AbstractStorage oriented implementations should open 
	 * the storage (DB, XML, File) and load the Meta data/defaulting 
	 * the <code>RowDefinition</code>.
     *
	 * @throws ModelException
	 *             if the <code>IDataBook</code> couldn't be opened, because of
	 *             empty storage unit name, empty <code>IDataSource</code>,
	 *             missing RowDefinition or failed <code>DataPage</code> creation
	 */
	public void open() throws ModelException;

	/**
	 * Closes the <code>IDataBook</code>. AbstractStorage oriented implementations should close 
	 * the storage (DB, XML, file).
	 */
	public void close();
	
	/**
	 * It inserts a new <code>IDataRow</code>. If bBeforeRow is true, it
	 * will be inserted before the selected <code>IDataRow</code>, otherwise after 
	 * that.
 	 * <br><br>
	 * Events:<br>
	 * It calls on all registered IControls, the saveEditing() method to store data first from the ui to the IDataBook.<br>
	 * Before the insert of the new row, the IDataBookListener.BEFORE_INSERTING event is raised.<br>
	 * Check the write back isolation level and consider to call saveAllDataBooks().
	 * After the insert of the new row, the IDataBookListener.AFTER_INSERTING event is raised.<br>
	 * After the selected row is changed, the IDataBookListener.AFTER_ROW_SELECTED event is raised.<br>
	 * After the selected row is changed, all registered IControls get called the notifyRepaint() method.<br>
	 * After the selected row is changed, all registered IControls of detail DataBooks get called 
	 * the notifyRepaint() method.
	 * 
	 * @param pBeforeRow
	 *            specifies if the <code>IDataRow</code> is inserted before or after the
	 *            selected <code>IDataRow</code>
	 * @return the index where the new <code>IDataRow</code> is placed            
	 * @throws ModelException 
	 *            if the <code>IDataRow</code> can not be inserted or the ReferenceDefinition is wrong
	 *            or if not all changes could be written to the storage 
	 *            (maybe implicit dataSource.storeAllDataBooks(), because of WritebackIsolationLevel
	 *             insert change the selected row implicit!)
	 */
	public int insert(boolean pBeforeRow) throws ModelException;
	
	
	/**
	 * It sets the selected row as UPDATING. That method will be implicit called from
	 * from the GUI controls, when the editing is started or when setValue(e)() is called. 
	 * DB implementations should Lock and refetch the selected row.
	 * <br><br>
	 * Events:<br>
	 * Before the update() is done, the IDataBookListener.BEFORE_UPDATING event is raised.<br>
	 * After the update() is done, the IDataBookListener.AFTER_UPDATING event is raised.<br>
	 * 
	 * @throws ModelException
	 * 				if problem with the locking in the storage happens.
	 */
	public void update() throws ModelException;
	
	/**
	 * It deletes the selected <code>DataRow</code>.
	 * <br><br>
	 * Events:<br>
	 * Before the delete of the selected row is done, the IDataBookListener.BEFORE_DELETING event is raised.
	 * <blockquote>
	 * Events, if Detail DataBooks with rows exist:<br>
	 * Before the delete each row is done, the IDataBookListener.BEFORE_DELETING event is raised.<br>
	 * After the delete each row is done, the IDataBookListener.AFTER_DELETING event is raised.<br>
	 * After the deletes all registered IControls get called the notifyRepaint() method.
	 * </blockquote>
	 * After the delete of the selected row is done, the IDataBookListener.AFTER_DELETING event is raised.<br>
	 * After the delete of the selected row is done, all registered IControls get called the notifyRepaint() method.
	 * <br><br>
	 * If the selected row was before the delete INSERTING, then the IDataBookListener.AFTER_ROW_SELECTED event is raised,
	 * because the selected row content wise changed.<br>
	 * 
	 * @throws ModelException 
	 *             if not all changes could be written to the storage 
	 *             (maybe implicit save(), because of WritebackIsolationLevel)
	 */
	public void delete() throws ModelException;
	
	/**
	 * It sets the selected row in master, that one or more details rows (from DetailDataBooks) are changed.
	 */
	public void notifyDetailChanged();
	
	/**
	 * It stores all changes in the selected <code>IDataRow</code>. 
     * <br><br>Events:<br>
     * It calls before the store on all registered IComponents the saveEditing() method, to set all
     * changes to the DataBook before the store operation.<br>
     * If the selected row is INSERTING the IDataBookListener.BEFORE_INSERTED and IDataBookListener.AFTER_INSERTED event will be raised.
     * If the selected row is UPDATING the IDataBookListener.BEFORE_UPDATED and IDataBookListener.AFTER_UPDATED event will be raised.
     * If the selected row is DELETING the IDataBookListener.BEFORE_DELETED and IDataBookListener.AFTER_DELETED event will be raised.
	 * After the store, all registered IControls get called the notifyRepaint() method.
     *  
	 * @throws ModelException 
	 *             if an <code>ModelException</code> happens during store operation.
	 */
	public void saveSelectedRow() throws ModelException;
	
	/**
	 * It restores the selected <code>IDataRow</code>, like the was before the changes. 
     * <br><br>Events:<br>
     * It calls before the restore on all registered IComponents the cancelEditing() method, to 
     * cancel the editing mode of the GUI control.<br>
	 * Before the restore, the IDataBookListener.BEFORE_RESTORE event is raised.<br>
	 * If the row isInserting then an IDataBookListener.AFTER_ROW_SELECTED will be raised<br>
	 * After the restore, the IDataBookListener.AFTER_RESTORE event is raised.<br>
	 * After the restore, all registered IControls get called the notifyRepaint() method.
	 * 
	 * @throws ModelException 
	 *             if an <code>ModelException</code> happens during undo the changes.
	 */
	public void restoreSelectedRow() throws ModelException;	
	
	/**
	 * It restores all <code>IDataRow</code>'s, like the was before the changes. 
     * <br><br>Events:<br>
     * It calls before the restore all, on all registered IComponents the cancelEditing() method, to 
     * cancel the editing mode of the GUI control.<br>
	 * It calls restore on the current selected row and then for all other changed rows.
	 * Before each restore, the setSelectedRow method with all events is called.
	 * <p/>
	 * That means that for each changed row following Events occur:
	 * <ul>
	 * <li>Before the restore, the IDataBookListener.BEFORE_RESTORE event is raised.</li>
	 * <li>If the row isInserting then an IDataBookListener.AFTER_ROW_SELECTED will be raised</li>
	 * <li>After the restore, the IDataBookListener.AFTER_RESTORE event is raised.</li>
	 * </ul>
	 * After the restore all rows, all registered IControls get called the notifyRepaint() method.
     *  
	 * @throws ModelException 
	 *             if an <code>ModelException</code> happens during undo the changes.
	 */
	public void restoreAllRows() throws ModelException;		

	/**
	 * Gets true, if the IDataBook is not yet initialized after a reload.
	 * This can be used by controls to optimize fetch on demand.
	 * 
	 * @return  true, if the IDataBook is not yet initialized after a reload.
	 */
	public boolean isOutOfSync();
	
	/**
	 * It refreshs all <code>IDataRow</code>'s in the IDataBook, like the was before the changes, get the data
	 * again from the storage
	 * It selects the same row (over primary key) if the SelectionMode == CURRENT_ROW in the IDataBook.
	 * It selects the first row if the SelectionMode == FIRST_ROW in the IDataBook.
     * <br><br>Events:<br>
     * It calls before the refresh, on all registered IComponents the cancelEditing() method, to 
     * cancel the editing mode of the GUI control.<br>
	 * Before the refresh, the IDataBookListener.BEFORE_RELOAD event is raised.<br>
	 * After the refresh, the IDataBookListener.AFTER_RELOAD event is raised.<br>
	 * After the refresh, all registered IControls get called the notifyRepaint() method.
     *  
	 * @throws ModelException 
	 *             if an <code>DataSourceException</code> happens during get the requested row.
	 */		
	public void reload() throws ModelException;	

	/**
	 * It refreshs all <code>IDataRow</code>'s in the IDataBook, like the was before the changes, get the data
	 * again from the storage and selects the row with the specified SelectionMode.
	 * It selects the same row (PK, ID) if the specified SelectionMode == CURRENT_ROW.
	 * It selects the first row if the specified SelectionMode == FIRST_ROW.
     * <br><br>Events:<br>
     * It calls before the refresh, on all registered IComponents the cancelEditing() method, to 
     * cancel the editing mode of the GUI control.<br>
	 * Before the refresh, the IDataBookListener.BEFORE_RELOAD event is raised.<br>
	 * After the refresh, the IDataBookListener.AFTER_RELOAD event is raised.<br>
	 * After the refresh, all registered IControls get called the notifyRepaint() method.
     *  
	 * @param pSelectionMode	the Selection mode to use
	 * @throws ModelException 
	 *             if an <code>DataSourceException</code> happens during get the requested row.
	 */		
	public void reload(SelectionMode pSelectionMode) throws ModelException;	
		
	/**
	 * Sets the filter for this <code>IDataBook</code>.<br>
	 * A filter consist of one ore more <code>ICondition</code>'s.
	 * It stores first all changes, and then sets the new filter.
	 * 
	 * @see javax.rad.model.condition.ICondition
	 * @param pFilter
	 *            the <code>Filter</code> with all <code>ICondition</code>'s.
	 * @throws ModelException 
	 *            if the <code>IDataBook</code> couldn't stored.
	 */
	public void setFilter(ICondition pFilter) throws ModelException;

	/**
	 * Returns the filter for this <code>IDataBook</code>.
	 * 
	 * @return the filter for this <code>IDataBook</code>.
	 */
	public ICondition getFilter();
	
	/**
	 * Sets the sort order for this <code>IDataBook</code>.<br>
	 * It stores first all changes, and then sets the new sort order, and 
	 * get the rows from the storage or sort it in memory.
	 * 
	 * @see javax.rad.model.SortDefinition
	 * @param pSort
	 *            the sort order for this <code>IDataBook</code>
	 * @throws ModelException 
	 *            if the <code>IDataBook</code> couldn't stored.
	 */
	public void setSort(SortDefinition pSort) throws ModelException;

	/**
	 * Returns the sort order for this <code>IDataBook</code>.
	 * 
	 * @return the sort order for this <code>IDataBook</code>.
	 */
	public SortDefinition getSort();
	
	/**
	 * Registered a new detail IDataBook to this master IDataBook.
	 * 
	 * @param pDataBook
	 * 				the detail DataBook to use
	 */
	public void addDetailDataBook(IDataBook pDataBook);
	
	/**
	 * Unregister a detail IDataBook from this master IDataBook.
	 * 
	 * @param pDataBook
	 * 				the detail IDataBook to use
	 */
	public void removeDetailDataBook(IDataBook pDataBook);
		
	/**
	 * Returns all detail <code>IDataBooks</code> to the master <code>IDataBook</code>.
	 * 
	 * @return all detail <code>IDataBooks</code> to the master <code>IDataBook</code>.
	 */
	public IDataBook[] getDetailDataBooks();
	
	/**
	 * Has to be called from the master, if the master changed, to notify all detail IDataBooks.
	 */
	public void notifyMasterChanged();	
	
    /**
     * Gets the EventHandler for before row selected event.
     * 
     * @return the EventHandler for before row selected event.
     */
	public DataBookHandler eventBeforeRowSelected();

    /**
     * Gets the EventHandler for after row selected event.
     * 
     * @return the EventHandler for after row selected event.
     */
	public DataBookHandler eventAfterRowSelected();
	
    /**
     * Gets the EventHandler for before inserting event.
     * 
     * @return the EventHandler for before inserting event.
     */
	public DataBookHandler eventBeforeInserting();
	
    /**
     * Gets the EventHandler for after inserting event.
     * 
     * @return the EventHandler for after inserting event.
     */
	public DataBookHandler eventAfterInserting();
	
    /**
     * Gets the EventHandler for before inserted event.
     * 
     * @return the EventHandler for before inserted event.
     */
	public DataBookHandler eventBeforeInserted();
	
    /**
     * Gets the EventHandler for after inserted event.
     * 
     * @return the EventHandler for after inserted event.
     */
	public DataBookHandler eventAfterInserted();
	
    /**
     * Gets the EventHandler for before updating event.
     * 
     * @return the EventHandler for before updating event.
     */
	public DataBookHandler eventBeforeUpdating();
	
    /**
     * Gets the EventHandler for after updating event.
     * 
     * @return the EventHandler for after updating event.
     */
	public DataBookHandler eventAfterUpdating();
	
    /**
     * Gets the EventHandler for before updated event.
     * 
     * @return the EventHandler for before updated event.
     */
	public DataBookHandler eventBeforeUpdated();
	
    /**
     * Gets the EventHandler for after updated event.
     * 
     * @return the EventHandler for after updated event.
     */
	public DataBookHandler eventAfterUpdated();
	
    /**
     * Gets the EventHandler for before deleting event.
     * 
     * @return the EventHandler for before deleting event.
     */
	public DataBookHandler eventBeforeDeleting();
	
    /**
     * Gets the EventHandler for after deleting event.
     * 
     * @return the EventHandler for after deleting event.
     */
	public DataBookHandler eventAfterDeleting();
	
    /**
     * Gets the EventHandler for before deleted event.
     * 
     * @return the EventHandler for before deleted event.
     */
	public DataBookHandler eventBeforeDeleted();
	
    /**
     * Gets the EventHandler for after deleted event.
     * 
     * @return the EventHandler for after deleted event.
     */
	public DataBookHandler eventAfterDeleted();
	
    /**
     * Gets the EventHandler for before restore event.
     * 
     * @return the EventHandler for before restore event.
     */
	public DataBookHandler eventBeforeRestore();
	
    /**
     * Gets the EventHandler for after restore event.
     * 
     * @return the EventHandler for after restore event.
     */
	public DataBookHandler eventAfterRestore();
	
    /**
     * Gets the EventHandler for before reload event.
     * 
     * @return the EventHandler for before reload event.
     */
	public DataBookHandler eventBeforeReload();
	
    /**
     * Gets the EventHandler for after reload event.
     * 
     * @return the EventHandler for after reload event.
     */
	public DataBookHandler eventAfterReload();
	
    /**
     * Gets the EventHandler for before column selected event.
     * 
     * @return the EventHandler for before column selected event.
     */
	public DataBookHandler eventBeforeColumnSelected();

    /**
     * Gets the EventHandler for after column selected event.
     * 
     * @return the EventHandler for after column selected event.
     */
	public DataBookHandler eventAfterColumnSelected();

	/**
	 * It stores all changes in the <code>IDataRow</code>'s to the <code>IDataSource</code>. 
     * <br><br>Events:<br>
     * It calls before the store on all registered IComponents the saveEditing() method, to set all
     * changes to the DataBook before the store operation.<br>
	 * Before the store on a row is done, the row will be selected and the usual XX_ROW_SELECTED 
	 * events will be raised. For each delete will be the same events raised, like at the
	 * saveSelectedRow() method.
	 * After the store, all registered IControls get called the notifyRepaint() method.
     *  
	 * @throws ModelException 
	 *             if an <code>ModelException</code> happens during store operation.
	 */	
	public void saveAllRows() throws ModelException;
	
	/**
	 * It deletes all <code>DataRow</code>s.
	 * <br><br>
	 * Before the delete on a row is done, the row will be selected and the usual XX_ROW_SELECTED 
	 * events will be raised. For each delete will be the same events raised, like at the
	 * delete() method.
	 * 
	 * @see #setSelectedRow(int)
	 * @see #delete()
	 * 
	 * @throws ModelException 
	 *             if not all changes could be written to the storage 
	 *             (maybe implicit save(), because of WritebackIsolationLevel)
	 */
	public void deleteAllDataRows() throws ModelException;	
	
	/**
	 * Removes the DataPage to the specified master DataRow or TreePath. 
	 * e.g. used to delete all DataPages to the master.
	 * 
	 * @param pMasterDataRow	the MasterDataRow to use.
	 * @param pTreePath			the TreePath to use.
	 * @throws ModelException	if the remove of the DataPage didn't worked out.
	 */
	public void removeDataPage(IDataRow pMasterDataRow, TreePath pTreePath) throws ModelException;
	
	/**
	 * Saves all rows in the current DataPage.
	 * 
	 * @throws ModelException if saveSelected() fails
	 */
	public void saveDataPage() throws ModelException;
	
	/**
	 * Returns the additional data row.
	 * The additional data row is only in memory. If it is set to visible, it is shown as first row.
	 * All controls will show the content of this data row, if selected.
	 * 
	 * @return the additional data row.
	 * @throws ModelException 
	 * 			  if the data book is not open.
	 */
	public IDataRow getAdditionalDataRow() throws ModelException;
	
	/**
	 * True, if the additional data row is visible.
	 * The data row is available in row number 0.
	 * @return true, if the additional data row is visible.
	 */
	public boolean isAdditionalDataRowVisible();
	
	/**
	 * Set true, if the additional data row is visible.
	 * The data row is available in row number 0.
	 * @param pVisible true, if the additional data row is visible.
	 */
	public void setAdditionalDataRowVisible(boolean pVisible);

	/**
	 * Sets the selected row index relatively to the current <code>IDataPage</code>.<br>
	 * This should be used from the tree controls to show which one is selected.
	 * The difference is due to a visible additional row.
	 * In case is is visible, setSelectedRow(1) is equal to setSelectedRowInPage(0).
	 * 
	 * @param pDataRowIndex
	 *            the selected row index relatively to the current <code>IDataPage</code>.
	 * @throws ModelException
	 *            if the row with the iDataRowIndex couldn't get from the
	 *            storage
	 *            or if the <code>IDataBook</code> isn't open or the master 
	 *            <code>IDataBook</code> has no selected row.
	 */
	public void setSelectedDataPageRow(int pDataRowIndex) throws ModelException;

	/**
	 * Returns the selected row index relatively to the current <code>IDataPage</code>.<br>
	 * This should be used from the tree controls to show which one is selected.
	 * The difference is due to a visible additional row.
	 * In case is is visible, setSelectedRow(1) is equal to setSelectedRowInPage(0).
	 * 
	 * @return the selected row index relatively to the current <code>IDataPage</code>.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
	 */
	public int getSelectedDataPageRow() throws ModelException;
	
} 	// IDataBook
