/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 19.10.2009 - [JR] - creation
 * 05.04.2014 - [JR] - #1001: don't change text if translation is disabled
 */
package javax.rad.genui.component;

import javax.rad.genui.UIComponent;
import javax.rad.genui.UIImage;
import javax.rad.genui.UIInsets;
import javax.rad.ui.IImage;
import javax.rad.ui.IInsets;
import javax.rad.ui.component.IActionComponent;
import javax.rad.ui.event.ActionHandler;
import javax.rad.ui.event.Key;

/**
 * Platform and technology independent action component.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF, ... .
 * 
 * @author Ren Jahn
 * 
 * @param <C> instance of IActionComponent
 */
public abstract class AbstractUIActionComponent<C extends IActionComponent> extends UIComponent<C> 
                                                                            implements IActionComponent 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the image. */
	private IImage image = null;
	
	/** the action component text. */
	private String sText = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
    /**
     * Creates a new instance of <code>AbstractUIActionComponent</code>.
     *
     * @param pActionComponent the {@link IActionComponent}.
     * @see IActionComponent
     */
	protected AbstractUIActionComponent(C pActionComponent)
	{
		super(pActionComponent);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface Implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public String getText()
    {
    	return sText;
    }

	/**
	 * {@inheritDoc}
	 */
	public void setText(String pText)
    {
		sText = pText;
		
    	uiResource.setText(translate(pText));
    }

	/**
	 * {@inheritDoc}
	 */
    public IImage getImage()
    {
    	return image;
    }
    
	/**
	 * {@inheritDoc}
	 */
    public void setImage(IImage pImage)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pImage instanceof UIImage)
		{
	    	uiResource.setImage(((UIImage)pImage).getUIResource());
		}
		else
		{
	    	uiResource.setImage(pImage);
		}
		
		image = pImage;
    }
    
	/**
	 * {@inheritDoc}
	 */
	public int getHorizontalAlignment()
    {
    	return uiResource.getHorizontalAlignment();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setHorizontalAlignment(int pHorizontalAlignment)
    {
    	uiResource.setHorizontalAlignment(pHorizontalAlignment);
    }

	/**
	 * {@inheritDoc}
	 */
	public int getVerticalAlignment()
    {
    	return uiResource.getVerticalAlignment();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setVerticalAlignment(int pVerticalAlignment)
    {
    	uiResource.setVerticalAlignment(pVerticalAlignment);
    }
	
	/**
	 * {@inheritDoc}
	 */
    public void setAccelerator(Key pKey)
    {
    	uiResource.setAccelerator(pKey);
    }

	/**
	 * {@inheritDoc}
	 */
    public Key getAccelerator()
    {
    	return uiResource.getAccelerator();
    }
    
	/**
	 * {@inheritDoc}
	 */
	public String getActionCommand()
    {
    	return uiResource.getActionCommand();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public void setActionCommand(String pActionCommand)
    {
    	uiResource.setActionCommand(pActionCommand);
    }

	/**
	 * {@inheritDoc}
	 */
	public ActionHandler eventAction()
	{
		return uiResource.eventAction();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IInsets getMargins()
	{
    	return uiResource.getMargins();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setMargins(IInsets pMargins)
	{
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pMargins instanceof UIInsets)
		{
	    	uiResource.setMargins(((UIInsets)pMargins).getUIResource());
		}
		else
		{
	    	uiResource.setMargins(pMargins);
		}
	}
	
	/**
	 * Sets the insets with primitive types.
	 * 
	 * @param pTop the top insets.
	 * @param pLeft the left insets.
	 * @param pBottom the bottom insets.
	 * @param pRight the right insets.
	 */
    public void setMargins(int pTop, int pLeft, int pBottom, int pRight)
    {
    	uiResource.setMargins(getFactory().createInsets(pTop, pLeft, pBottom, pRight));
    }

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void updateTranslation()
	{
		boolean bChanged = isTranslationChanged();
		
		super.updateTranslation();
		
		if (bTranslate && bChanged)
		{
			uiResource.setText(translate(sText));
		}
	}
	
}	// AbstractUIButton
