/*
 * Copyright 2013 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 09.04.2014 - [HM] - creation
 */
package javax.rad.genui;

import java.util.WeakHashMap;

/**
 * The <code>AbstractUIFactoryResource</code> holds the resource per factory. It creates a resource
 * copy if the resource was initialized as static resource.
 * 
 * @author Martin Handsteiner
 *
 * @param <UI> the resource type
 */
public abstract class AbstractUIFactoryResource<UI>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The resource. */
	private WeakHashMap<Class, UI> whmpResources = new WeakHashMap<Class, UI>();

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>AbstractUIFactoryResource</code>.
	 * 
	 * @param pResource the resource
	 */
	protected AbstractUIFactoryResource(UI pResource)
	{
		synchronized (whmpResources)
		{
			whmpResources.put(UIFactoryManager.getFactory().getClass(), pResource);
		}
	}
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Overwritten methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the UI resource corresponding with this wrapper object.
	 * 
	 * @return the original resource
	 */
	public final UI getUIResource()
	{
		UI res;
		
		synchronized (whmpResources)
		{
			res = whmpResources.get(UIFactoryManager.getFactory().getClass());
		}
		
		if (res == null)
		{
			UI uiResource = whmpResources.entrySet().iterator().next().getValue();
			
			res = UIFactoryManager.cloneResource(uiResource);
			
			if (res == null || res == uiResource)
			{
				throw new RuntimeException("Unsupported object type: " + uiResource.getClass().getName());
			}
			
			synchronized (whmpResources)
			{
				whmpResources.put(UIFactoryManager.getFactory().getClass(), res);
			}
		}
		
		return res;
	}
	
}	// AbstractUIFactoryResource
