/*
 * Copyright 2014 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 27.10.2014 - [JR] - creation
 */
package com.sibvisions.util.type;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;

/**
 * The <code>HttpUtil</code> is a utility class for http communication.
 * 
 * @author Ren Jahn
 */
public final class HttpUtil
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Invisible constructor because <code>HttpUtil</code> is a utility
     * class.
     */
    private HttpUtil()
    {
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Sends an empty post request.
     * 
     * @param pURL the URL
     * @return the response stream
     * @throws Exception if communication failed
     */
    public static InputStream post(String pURL) throws Exception
    {
        return post(pURL, null, null);
    }
    
    /**
     * Sends a post request with given data.
     * 
     * @param pURL the URL
     * @param pStream the content to send
     * @return the response stream
     * @throws Exception if communication failed
     */
    public static InputStream post(String pURL, InputStream pStream) throws Exception
    {
        return post(pURL, pStream, null);
    }
    
    /**
     * Sends a post request with given file.
     * 
     * @param pURL the URL
     * @param pFile the file
     * @return the response stream
     * @throws Exception if communication failed
     */
    public static InputStream post(String pURL, File pFile) throws Exception
    {
        FileInputStream fis = new FileInputStream(pFile);
        
        try
        {
            return post(pURL, fis, pFile.getName());
        }
        finally
        {
            CommonUtil.close(fis);
        }
    }
    
    /**
     * Sends a post request with given content.
     * 
     * @param pURL the URL
     * @param pStream the content to send
     * @param pFileName the file name
     * @return the response stream
     * @throws Exception if communication failed
     */
    public static InputStream post(String pURL, InputStream pStream, String pFileName) throws Exception
    {
        URL url = new URL(pURL);
        
        URLConnection ucon = url.openConnection(); 
        ucon.setDoOutput(pStream != null);
        ucon.setDoInput(true);
        ucon.setUseCaches(false);
        
        if (pStream != null)
        {
            ucon.setRequestProperty("Content-Type", "application/octet-stream");
            
            if (pFileName != null)
            {
                ucon.setRequestProperty("Content-Disposition", "attachment; filename=\"" + pFileName + "\";");
            }

            FileUtil.copy(pStream, ucon.getOutputStream());
        }

        return ucon.getInputStream(); 
    }
    
    /**
     * Sends a get request.
     * 
     * @param pURL the URL
     * @return the response stream
     * @throws Exception if communication failed
     */
    public static InputStream get(String pURL) throws Exception
    {
        URL url = new URL(pURL);
        
        URLConnection ucon = url.openConnection(); 
        ucon.setDoOutput(false);
        ucon.setDoInput(true);
        ucon.setUseCaches(false);

        return ucon.getInputStream(); 
    }
    
}   // HttpUtil
