/*
 * Copyright 2011 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 11.12.2011 - [JR] - creation
 * 23.12.2012 - [JR] - #534: logging
 */
package com.sibvisions.rad.server.http.rest;

import org.restlet.Request;
import org.restlet.Response;
import org.restlet.security.Verifier;

import com.sibvisions.rad.server.DirectServerSession;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>SecurityManagerVerifier</code> uses the security mangaer of an application to authenticate a user.
 * 
 * @author Ren Jahn
 */
public class SecurityManagerVerifier implements Verifier
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /** the connection to the adapter/application. */
    private RESTAdapter adapter;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Creates a new instance of <code>SecurityManagerVerifier</code>.
     * 
     * @param pAdapter the application/REST adapter
     */
    public SecurityManagerVerifier(RESTAdapter pAdapter)
    {
        adapter = pAdapter;
    }
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
    /**
     * {@inheritDoc}
     */
    public int verify(Request pRequest, Response pResponse) 
    {
		if (pRequest.getChallengeResponse() == null) 
        {
            return RESULT_MISSING;
        } 
        else 
        {
        	String sApplicationName = (String)pRequest.getAttributes().get(RESTAdapter.PARAM_APP_NAME);
        	
    		if (sApplicationName == null)
    		{
    			return RESULT_MISSING;
    		}
    		
        	String sIdentifier = getIdentifier(pRequest, pResponse);
            String sSecret = new String(getSecret(pRequest, pResponse));
            
            DirectServerSession session = null;
            
            try
            {
            	session = DirectServerSession.createMasterSession(sApplicationName, sIdentifier, sSecret);

            	pRequest.getClientInfo().setUser(new LifeCycleConnector(adapter, session, (String)pRequest.getAttributes().get(RESTAdapter.PARAM_LCO_CLASS)));
            	
            	return RESULT_VALID;
            }
            catch (Throwable th)
            {
            	LoggerFactory.getInstance(SecurityManagerVerifier.class).error(th);
            	
            	if (session != null)
            	{
	            	try
	            	{
	            		session.destroy();
	            	}
	            	catch (Throwable thr)
	            	{
	            		//nothing to be done
	            	}
            	}
            	
            	return RESULT_INVALID;
            }
        }
    }

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Returns the user identifier.
     * 
     * @param pRequest The request to inspect.
     * @param pResponse The response to inspect.
     * @return The user identifier.
     */
    protected String getIdentifier(Request pRequest, Response pResponse) 
    {
        return pRequest.getChallengeResponse().getIdentifier();
    }

    /**
     * Returns the secret provided by the user.
     * 
     * @param pRequest The request to inspect.
     * @param pResponse The response to inspect.
     * @return The secret provided by the user.
     */
    protected char[] getSecret(Request pRequest, Response pResponse) 
    {
        return pRequest.getChallengeResponse().getSecret();
    }    

}	// SecurityManagerVerifier
