/*
 * Copyright 2013 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 16.06.2013 - [JR] - #673: creation
 */
package com.sibvisions.rad.server.config;

import java.io.File;
import java.util.List;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.xml.XmlNode;

/**
 * The <code>AppSetting</code> class encapsulates the access
 * to app settings. The settings configure application locations. 
 * 
 * @author Ren Jahn
 */
public final class AppSettings extends Zone
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>AppSettings</code>.
	 * 
	 * @param pDirectory the config directory
	 * @throws Exception if settings are not valid
	 */
	AppSettings(File pDirectory) throws Exception
	{
		super(pDirectory, "apps.xml");
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void update() throws Exception
	{
		if (isValid())
		{
			super.update();
		}
		else
		{
			setContent(new XmlNode("apps"));
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets all configured application directories.
	 * 
	 * @return the appliciation dir location
	 */
	public List<String> getAppLocations()
	{
		try
		{
			List<XmlNode> liNodes = getNodes("/apps/dir");
	
			if (liNodes != null)
			{
				ArrayUtil<String> auLocs = new ArrayUtil<String>();
				
				for (int i = 0, anz = liNodes.size(); i < anz; i++)
				{
					auLocs.add(replace(liNodes.get(i).getNodeValue("/location")));
				}
				
				return auLocs;
			}
		}
		catch (Exception e)
		{
			//nothing to be done
		}
		
		return null;
	}
	
	/**
	 * Replaces all system property placeholders with the current value of the system property.
	 * A placeholder is defined as ${name}.
	 * 
	 * @param pValue the configuration value
	 * @return the value with replaced system properties
	 */
	private String replace(String pValue)
	{
		if (pValue != null)
		{
			StringBuilder sbValue = new StringBuilder(pValue);
			
			String sProp;
			
			char ch;
			
			for (int i = 0; i < sbValue.length() - 1; i++)
			{
				ch = sbValue.charAt(i);
				
				if (ch == '$' && sbValue.charAt(i + 1) == '{')
				{
					for (int j = i + 1, anz = sbValue.length(); j < anz; j++)
					{
						if (sbValue.charAt(j) == '}')
						{
							sProp = System.getProperty(sbValue.substring(i + 2, j), "");
							
							sbValue.replace(i, j + 1, sProp);
							
							i = j + 1;
							j = anz;
						}
					}
				}
			}
			
			return sbValue.toString();
		}
		else
		{
			return null;
		}
	}

}	// AppSettings
