/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 20.01.2010 - [HM] - creation
 */
package com.sibvisions.rad.remote.serializer;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.sql.Timestamp;
import java.util.Date;

import com.sibvisions.rad.remote.UniversalSerializer;

/**
 * The serializer for {@link Date}.
 *  
 * @author Martin Handsteiner
 */
public class DateSerializer implements ITypeSerializer<Date>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** <code>java.sql.Timestamp</code> type. */
	private static final int TYPE_TIMESTAMP = 18;

	/** <code>java.sql.Timestamp</code> type. */
	private static final int TYPE_TIMESTAMP_NONANOS = 19;
	
	/** Caches the timestamp. */
	private static Timestamp[] timestampCache = new Timestamp[256];
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface Implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public Class<Date> getTypeClass()
	{
		return Date.class;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int getMinValue()
	{
		return TYPE_TIMESTAMP;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int getMaxValue()
	{
		return TYPE_TIMESTAMP_NONANOS;
	}

	/**
	 * {@inheritDoc}
	 */
	public Date read(UniversalSerializer pSerializer, DataInputStream pIn, int pTypeValue, TypeCache pCache) throws Exception
	{
		long time = pIn.readLong();
		
		int index = (int)(time >>> 8);
		index ^= (index >>> 20) ^ (index >>> 12);
		index ^= (index >>> 7) ^ (index >>> 4);
		index &= 0xff;
		
		Timestamp timestamp = timestampCache[index];
		
		if (timestamp == null || time != timestamp.getTime())
		{
			timestamp = new Timestamp(time);
			
			timestampCache[index] = timestamp;
		}
		
		//not necessary, but we read if the write method implements that case!
		if (pTypeValue == TYPE_TIMESTAMP)
		{
			pIn.readInt();
		}
		
		return timestamp;
	}

	/**
	 * {@inheritDoc}
	 */
	public void write(UniversalSerializer pSerializer, DataOutputStream pOut, Date pObject, TypeCache pCache) throws Exception
	{
		pOut.writeByte(TYPE_TIMESTAMP_NONANOS);
		pOut.writeLong(pObject.getTime());
	}
	
}	// DateSerializer
