/*
 * Created on 2013.09.02.
 * 
 * Copyright 2013 progos.hu All rights reserved. PROGOS
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * Author: arpadtamasi
 * $URL$
 * $Rev$
 * $Author$
 * $Date$
 * $Id$
 *
 */

package com.samebug.notifier.logback;

import java.net.MalformedURLException;
import java.util.Date;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.SynchronousQueue;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.AppenderBase;

import com.samebug.notifier.ConfigurationFactory;
import com.samebug.notifier.IReporter;
import com.samebug.notifier.StreamReporter;
import com.samebug.notifier.core.IConfiguration;
import com.samebug.notifier.core.IResponse;
import com.samebug.notifier.core.SamebugNotifier;
import com.samebug.notifier.core.exceptions.BadServerAddress;
import com.samebug.notifier.core.exceptions.NotifierException;
import com.samebug.notifier.core.exceptions.RecorderError;
import com.samebug.notifier.core.exceptions.UnsupportedUTF8;
import com.samebug.notifier.core.exceptions.UrlEncodingError;

public class SamebugAppender extends AppenderBase<ILoggingEvent> implements Runnable {
    private int queueSize = 0;

    private BlockingQueue<ILoggingEvent> queue;

    private Future<?> task;

    private final SamebugNotifier notifier;
    private final IReporter reporter;

    /**
     * Constructs a new appender.
     * 
     * @throws BadServerAddress
     * @throws MalformedURLException
     * @throws UrlEncodingError
     * @throws UnsupportedUTF8
     */
    public SamebugAppender() {
        this(ConfigurationFactory.fromDefault());
    }

    public SamebugAppender(final IConfiguration config) {
        super();
        reporter = new StreamReporter(config);
        notifier = new SamebugNotifier(config);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void start() {
        if (isStarted()) { return; }

        int errorCount = 0;

        if (this.queueSize < 0) {
            errorCount++;
            addError("Queue size must be non-negative");
        }

        if (errorCount == 0) {
            this.queue = newBlockingQueue(this.queueSize);
            this.task = getContext().getExecutorService().submit(this);
            super.start();
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void stop() {
        if (!isStarted()) { return; }
        this.task.cancel(true);

        super.stop();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void append(final ILoggingEvent event) {
        if (event == null || event.getThrowableProxy() == null || !isStarted()) { return; }
        try {
            this.queue.put(event);
        } catch (final InterruptedException e) {
            // SKIP
        }
    }

    /**
     * {@inheritDoc}
     */
    public final void run() {
        try {
            while (!Thread.currentThread().isInterrupted()) {
                final ILoggingEvent event = this.queue.take();

                try {
                    IResponse response = notifier.notify(event.getMessage(), new LogbackThrowableProxy(event.getThrowableProxy()), new Date());
                    reporter.handleResponse(response);
                } catch (final RecorderError e) {
                    reporter.handleRecorderError(e);
                } catch (final NotifierException e) {
                    reporter.handleNotifierException(e);
                }
            }
        } catch (final InterruptedException ex) {
            // EXIT
        }

    }

    /**
     * Creates a blocking queue that will be used to hold logging events until
     * they can be delivered to the remote receiver.
     * <p>
     * The default implementation creates a (bounded) {@link ArrayBlockingQueue}
     * for positive queue sizes. Otherwise it creates a {@link SynchronousQueue}.
     * <p>
     * This method is exposed primarily to support instrumentation for unit
     * testing.
     * 
     * @param size
     *            size of the queue
     * @return
     */
    BlockingQueue<ILoggingEvent> newBlockingQueue(final int size) {
        return size <= 0 ? new SynchronousQueue<ILoggingEvent>() : new ArrayBlockingQueue<ILoggingEvent>(size);
    }

    /**
     * The <b>queueSize</b> property takes a non-negative integer representing
     * the number of logging events to retain for delivery to the remote
     * receiver. When the queue size is zero, event delivery to the remote
     * receiver is synchronous. When the queue size is greater than zero, the
     * {@link #append(Object)} method returns immediately after enqueing the
     * event, assuming that there is space available in the queue. Using a
     * non-zero queue length can improve performance by eliminating delays
     * caused by transient network delays. If the queue is full when the
     * {@link #append(Object)} method is called, the event is summarily and
     * silently dropped.
     * 
     * @param queueSize
     *            the queue size to set.
     */
    public void setQueueSize(final int queueSize) {
        this.queueSize = queueSize;
    }

    /**
     * Returns the value of the <b>queueSize</b> property.
     */
    public int getQueueSize() {
        return this.queueSize;
    }
}