/*
 * Created on 2013.09.18.
 * 
 * Copyright 2013 progos.hu All rights reserved. PROGOS
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * Author: arpadtamasi
 * $URL$
 * $Rev$
 * $Author$
 * $Date$
 * $Id$
 *
 */

package com.samebug.notifier;

import java.lang.Thread.UncaughtExceptionHandler;

import com.samebug.notifier.core.IConfiguration;
import com.samebug.notifier.core.INotifier;
import com.samebug.notifier.core.SamebugNotifier;
import com.samebug.notifier.core.exceptions.NotifierException;
import com.samebug.notifier.core.exceptions.RecorderError;
import com.samebug.notifier.core.exceptions.SamebugCapturedException;

/**
 * An uncaught exception handler, which sends the exceptions to Samebug.
 * <p>
 * You can chain an other uncaughtExceptionHandler to this with
 * {@link #setDefaultHandler(UncaughtExceptionHandler)}, which will be called
 * after sending the exception to the server. If you want this handler to
 * rethrow the exceptions (e.g. because you want them to be logged), use a
 * {@link DefaultUncaughtExceptionHandler} instance.
 * <p>
 * 
 * @author poroszd
 * 
 */
public class SamebugUncaughtExceptionHandler implements UncaughtExceptionHandler {
    private final IReporter reporter;
    private final INotifier notifier;
    private final UncaughtExceptionHandler defaultHandler;

    public SamebugUncaughtExceptionHandler(final IConfiguration config, final UncaughtExceptionHandler handler) {
        assert (config != null);
        assert (handler != null);
        this.reporter = new StreamReporter(config);
        this.notifier = new SamebugNotifier(config);
        this.defaultHandler = handler;
    }

    public void uncaughtException(final Thread t, final Throwable e) {
        try {
            reporter.handleResponse(notifier.notify("Uncaught exception", e));
        } catch (final RecorderError re) {
            reporter.handleRecorderError(re);
        } catch (final NotifierException ne) {
            reporter.handleNotifierException(ne);
        }
        defaultHandler.uncaughtException(t, e);

    }
}

class DefaultUncaughtExceptionHandler implements UncaughtExceptionHandler {
    public void uncaughtException(final Thread t, final Throwable e) {
        throw new SamebugCapturedException("Unhandled exception in thread " + t.getId(), e);
    }
}
