package com.samebug.notifier;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Properties;
import java.util.UUID;

import com.samebug.notifier.core.IConfiguration;
import com.samebug.notifier.core.exceptions.BadAppKey;
import com.samebug.notifier.core.exceptions.BadConfigFile;
import com.samebug.notifier.core.exceptions.BadDeveloperKey;
import com.samebug.notifier.core.exceptions.BadHelpMode;
import com.samebug.notifier.core.exceptions.BadServerAddress;
import com.samebug.notifier.core.exceptions.MultipleConfigFileException;
import com.samebug.notifier.core.exceptions.NoConfigFileException;

/**
 * Configuration parameters of samebug notifier.
 * <p>
 * It contains the following parameters:
 * <ul>
 * <li>appKey: the application key, mandatory</li>
 * <li>version: application version, default is <code>null</code></li>
 * <li>serverURL: url of the recorder, default is
 * <code>http://recorder.samebug.io</code></li>
 * </ul>
 * 
 * @author poroszd
 * 
 */
public class ConfigurationFactory {
    private static final String SAMEBUG_USER_CONFIG = "/.samebug/config";
    public static final String K_APP_KEY = "samebug.key";
    public static final String K_VERSION = "samebug.version";
    public static final String K_RECORDER_URL = "samebug.server";
    public static final String K_DEBUG = "samebug.debug";
    public static final String K_UI_URL = "samebug.ui";
    public static final String K_DEVELOPER_KEY = "samebug.developer.key";
    public static final String K_HELP_MODE = "samebug.help";

    public static final String DEF_APP_KEY = null;
    public static final String DEF_VERSION = null;
    public static final String DEF_RECORDER_URL = "http://recorder.samebug.io";
    public static final String DEF_DEBUG = "false";
    public static final String DEF_UI_URL = "http://beta.samebug.io";
    public static final String DEF_DEVELOPER_KEY = null;
    public static final String DEF_HELP_MODE = "0";

    public static final String DEFAULT_PROPERTY_RESOURCE = "samebug.properties";

    public static UUID parseAppKey(final String appKey) throws BadAppKey {
        if (appKey == null) throw new BadAppKey("Application key is missing!");
        try {
            return UUID.fromString(appKey);
        } catch (final IllegalArgumentException e) {
            throw new BadAppKey(appKey + " is not parsable as application key", e);
        }
    }

    public static String parseVersion(final String version) {
        return version;
    }

    public static URL parseRecorderURL(final String serverAddress) throws BadServerAddress {
        if (serverAddress == null) throw new BadServerAddress("Recorder address must not be null.");
        try {
            return new URI(serverAddress).toURL();
        } catch (final URISyntaxException e) {
            throw new BadServerAddress(serverAddress + " is not a valid URI", e);
        } catch (final IllegalArgumentException e) {
            throw new BadServerAddress(serverAddress + " is not an absolute URI", e);
        } catch (final MalformedURLException e) {
            throw new BadServerAddress(serverAddress + " is not a valid URL", e);
        }
    }

    public static URL parseUIURL(final String serverAddress) throws BadServerAddress {
        if (serverAddress == null) throw new BadServerAddress("UI address must not be null.");
        try {
            return new URI(serverAddress).toURL();
        } catch (final URISyntaxException e) {
            throw new BadServerAddress(serverAddress + " is not a valid URI", e);
        } catch (final IllegalArgumentException e) {
            throw new BadServerAddress(serverAddress + " is not an absolute URI", e);
        } catch (final MalformedURLException e) {
            throw new BadServerAddress(serverAddress + " is not a valid URL", e);
        }
    }

    public static boolean parseDebug(final String debug) {
        return (!debug.equals("false"));
    }

    public static UUID parseDeveloperKey(final String userKey) throws BadDeveloperKey {
        if (userKey == null) return null;
        try {
            return UUID.fromString(userKey);
        } catch (final IllegalArgumentException e) {
            throw new BadDeveloperKey(userKey + " is not parsable as user key", e);
        }
    }

    public static int parseHelpMode(final String helpMode) throws BadHelpMode {
        if (helpMode == null) return 0;
        try {
            return Integer.parseInt(helpMode);
        } catch (final NumberFormatException e) {
            throw new BadHelpMode(helpMode + " is not parsable as help mode", e);
        }
    }

    /**
     * Factory method of the default configuration.
     * <p>
     * This method will build a configuration based on values taken from many
     * location. Each value is assigned in this order:
     * <ol>
     * <li>command line property parameter (e.g.
     * -Dsamebug.key=0092029d-6f46-46a8-a65c-3e2dc22e5f0a)</li>
     * <li>content of the file or resource specified on command line
     * (-Dsamebug.config.file=/home/poroszd/my_app/samebug.properties or
     * -Dsamebug.config.resource=my_app.properties), or lack of these, the
     * content of the default samebug property resource (i.e. samebug.properties
     * in the classpath)</li>
     * <li>the default value of the property, if not mandatory</li>
     * </ol>
     * 
     * @return a configuration
     */
    public static IConfiguration fromDefault() throws NoConfigFileException, MultipleConfigFileException, BadConfigFile, BadHelpMode, BadServerAddress, BadDeveloperKey {
        String resource = System.getProperty("samebug.config.resource");
        String file = System.getProperty("samebug.config.file");
        if (resource == null && file == null) return fromPropertyResource(DEFAULT_PROPERTY_RESOURCE);
        if (resource != null && file == null) return fromPropertyResource(resource);
        if (resource == null && file != null) return fromPropertyFile(file);
        throw new MultipleConfigFileException("You specified more than one property file for samebug:\nsamebug.config.file=" + file + "\nsamebug.config.resource=" + resource);
    }

    /**
     * Factory method to create a configuration based on the content of a
     * resource property file.
     * 
     * @param resourceName
     *            the name of the resource
     * @return a configuration
     */
    public static IConfiguration fromPropertyResource(final String resourceName) throws NoConfigFileException, MultipleConfigFileException, BadConfigFile, BadServerAddress, BadHelpMode {
        try {
            Enumeration<URL> urls = ConfigurationFactory.class.getClassLoader().getResources(resourceName);
            if (!urls.hasMoreElements()) throw new NoConfigFileException("No configuration resource found with name " + resourceName);
            URL onlyConfig = urls.nextElement();
            if (urls.hasMoreElements()) throw new MultipleConfigFileException("Multiple configuration resource found in the classpath with name " + resourceName);
            InputStream propStream = onlyConfig.openStream();
            Properties props = new Properties();
            try {
                props.load(propStream);
                return fromProperties(props);
            } catch (final IOException e) {
                throw new BadConfigFile("Configuration resource " + resourceName + " is not a valid property file", e);
            } catch (final IllegalArgumentException e) {
                throw new BadConfigFile("Configuration resource " + resourceName + " has malformed unicode escape sequences", e);
            }
        } catch (final IOException e) {
            throw new NoConfigFileException("The configuration resource " + resourceName + " could not be opened", e);
        }
    }

    /**
     * Factory method to create a configuration based on the content of a
     * property file, given with its absolute path.
     * 
     * @param fileName
     *            the name of the file
     * @return a configuration
     */
    public static IConfiguration fromPropertyFile(final String fileName) throws NoConfigFileException, MultipleConfigFileException, BadConfigFile, BadServerAddress, BadHelpMode {
        Properties props = new Properties();
        InputStream propFile;
        try {
            propFile = new FileInputStream(fileName);
        } catch (FileNotFoundException e) {
            throw new NoConfigFileException("No configuration file found: " + fileName, e);
        }
        try {
            props.load(propFile);
            return fromProperties(props);
        } catch (final IOException e) {
            throw new BadConfigFile("Configuration file " + fileName + " is not a valid property file", e);
        } catch (final IllegalArgumentException e) {
            throw new BadConfigFile("Configuration file " + fileName + " has malformed unicode escape sequences", e);
        }
    }

    /**
     * Construct a configuration with the specified application key.
     * <p>
     * The other parameters are left on default.
     * 
     * @param appKey
     *            the key of your registered Samebug application, as found on
     *            the documentation site of your app. Must not be
     *            <code>null</code>
     * @throws BadHelpMode
     *             when appKey is not parsable as a UUID, or is
     *             <code>null</code>
     */
    public static IConfiguration fromAppKey(final String appKey) throws BadAppKey {
        Properties props = new Properties();
        props.put(K_APP_KEY, appKey);
        return fromProperties(props);
    }

    public static IConfiguration fromProperties(final Properties props) {
        props.putAll(System.getProperties());

        String appKey = props.getProperty(K_APP_KEY, DEF_APP_KEY);
        String version = props.getProperty(K_VERSION, DEF_VERSION);
        String recorderAddress = props.getProperty(K_RECORDER_URL, DEF_RECORDER_URL);
        String debug = props.getProperty(K_DEBUG, DEF_DEBUG);
        String developerKey = DEF_DEVELOPER_KEY;
        String uiAddress = props.getProperty(K_UI_URL, DEF_UI_URL);
        String helpMode = props.getProperty(K_HELP_MODE, DEF_HELP_MODE);

        try {
            File userHome = new File(System.getProperty("user.home"));
            Properties userProps = new Properties();
            InputStream userPropFile = new FileInputStream(userHome + SAMEBUG_USER_CONFIG);
            userProps.load(userPropFile);
            developerKey = userProps.getProperty(K_DEVELOPER_KEY, DEF_DEVELOPER_KEY);
        } catch (FileNotFoundException e) {
            // leave userKey null
        } catch (IOException e) {
            // leave userKey null
        }

        return new Configuration(parseAppKey(appKey), parseVersion(version), parseRecorderURL(recorderAddress), parseDebug(debug), parseDeveloperKey(developerKey), parseUIURL(uiAddress),
                parseHelpMode(helpMode));
    }
}

class Configuration implements IConfiguration {
    private final UUID appKey;
    private final String version;
    private final URL recorderURL;
    private final boolean debug;
    private final UUID developerKey;
    private final URL uiURL;
    private final int helpMode;

    /**
     * Construct a configuration with every parameter specified.
     * 
     * @param appKey
     *            Application key in Samebug. Must not be <code>null</code>!
     * @param version
     *            Your application version. Arbitrary string.
     * @param recorderAddress
     *            The absolute url of a Samebug Recorder. Must not be
     *            <code>null</code>!
     * @param debug
     *            Print debug messages
     */
    public Configuration(final UUID appKey, final String version, final URL recorderAddress, final boolean debug, final UUID userKey, final URL uiAddress, final int helpMode) {
        this.appKey = appKey;
        this.version = version;
        this.recorderURL = recorderAddress;
        this.debug = debug;
        this.developerKey = userKey;
        this.uiURL = uiAddress;
        this.helpMode = helpMode;
        assert (this.appKey != null);
        assert (this.recorderURL != null);
        assert (this.uiURL != null);
    }

    public UUID getAppKey() {
        return appKey;
    }

    public String getVersion() {
        return version;
    }

    public URL getRecorderURL() {
        return recorderURL;
    }

    public boolean getDebug() {
        return debug;
    }

    public UUID getDeveloperKey() {
        return developerKey;
    }

    public URL getUIURL() {
        return uiURL;
    }

    public int getHelpMode() {
        return helpMode;
    }

    public String toString() {
        return ConfigurationFactory.K_APP_KEY + " = " + appKey + "\n" + ConfigurationFactory.K_VERSION + " = " + version + "\n" + ConfigurationFactory.K_RECORDER_URL + " = " + recorderURL + "\n"
                + ConfigurationFactory.K_DEBUG + " = " + debug + "\n" + ConfigurationFactory.K_DEVELOPER_KEY + " = " + developerKey;
    }
}
