package com.rbrooks.indefinitepagerindicator

import android.content.Context
import android.content.res.Resources
import android.graphics.Canvas
import android.graphics.Paint
import android.support.annotation.ColorInt
import android.support.v4.view.ViewPager
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.RecyclerView
import android.util.AttributeSet
import android.util.DisplayMetrics
import android.view.View
import android.view.animation.DecelerateInterpolator

class IndefinitePagerIndicator @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyle: Int = 0) : View(context, attrs, defStyle), ViewPager.OnPageChangeListener {

    companion object {

        private val DEFAULT_DOT_COUNT = 5
        private val DEFAULT_FADING_DOT_COUNT = 1
        private val DEFAULT_DOT_RADIUS_DP = 4
        private val DEFAULT_SELECTED_DOT_RADIUS_DP = 5.5f
        // Measured outside of first dot to outside of next dot: O<->O
        private val DEFAULT_DOT_SEPARATION_DISTANCE_DP = 10

        private fun dpToPx(dp: Float, resources: Resources): Int =
                (dp * ((resources.displayMetrics.densityDpi / DisplayMetrics.DENSITY_DEFAULT).toFloat())).toInt()

    }

    private var recyclerView: RecyclerView? = null
    private var viewPager: ViewPager? = null
    private var internalRecyclerScrollListener: InternalRecyclerScrollListener? = null
    private val interpolator = DecelerateInterpolator()

    private var dotCount = DEFAULT_DOT_COUNT
    private var fadingDotCount = DEFAULT_FADING_DOT_COUNT
    private var selectedDotRadiusPx = dpToPx(DEFAULT_SELECTED_DOT_RADIUS_DP, resources)
    private var dotRadiusPx = dpToPx(DEFAULT_DOT_RADIUS_DP.toFloat(), resources)
    private var dotSeparationDistancePx = dpToPx(DEFAULT_DOT_SEPARATION_DISTANCE_DP.toFloat(), resources)

    @ColorInt
    private var dotColor: Int = resources.getColor(R.color.default_dot_color)
    @ColorInt
    private var selectedDotColor: Int = resources.getColor(R.color.default_selected_dot_color)
    private val selectedDotPaint = Paint()
    private val dotPaint = Paint()

    /**
     * The current pager position. Used to draw the selected dot if different size/color.
     */
    private var selectedItemPosition: Int = 0

    /**
     * A temporary value used to reflect changes/transition from one selected item to the next.
     */
    private var intermediateSelectedItemPosition: Int = 0

    /**
     * The scroll percentage of the viewpager or recyclerview.
     * Used for moving the dots/scaling the fading dots.
      */
    private var offsetPercent: Float = 0f

    init {
        attrs?.let {
            val typedArray = context.theme.obtainStyledAttributes(attrs, R.styleable.IndefinitePagerIndicator, 0, 0)
            dotCount = typedArray.getInteger(R.styleable.IndefinitePagerIndicator_dotCount, DEFAULT_DOT_COUNT)
            fadingDotCount = typedArray.getInt(R.styleable.IndefinitePagerIndicator_fadingDotCount, DEFAULT_FADING_DOT_COUNT)
            dotRadiusPx = typedArray.getDimensionPixelSize(R.styleable.IndefinitePagerIndicator_dotRadius, dotRadiusPx)
            selectedDotRadiusPx = typedArray.getDimensionPixelSize(R.styleable.IndefinitePagerIndicator_selectedDotRadius, selectedDotRadiusPx)
            dotColor = typedArray.getColor(R.styleable.IndefinitePagerIndicator_dotColor, dotColor)
            selectedDotColor = typedArray.getColor(R.styleable.IndefinitePagerIndicator_selectedDotColor, selectedDotColor)
            dotSeparationDistancePx = typedArray.getDimensionPixelSize(R.styleable.IndefinitePagerIndicator_dotSeparation, dotSeparationDistancePx)
        }

        selectedDotPaint.style = Paint.Style.FILL
        selectedDotPaint.color = selectedDotColor
        dotPaint.style = Paint.Style.FILL
        dotPaint.color = dotColor
    }

    /**
     * Iterate over the total pager item count and draw every dot based on position.
     *
     * Helper methods - getDotXCoordinate(Int) & getRadius(Int)
     * will return values outside the calculated width or with an invalid radius
     * if the dot is not to be drawn.
     *
     * TODO: "Instagram style" drawing where all dots are drawn at once.
     */
    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        for (i in 0 until getPagerItemCount()) {
            val xCoordinate = getDotXCoordinate(i)
            val normalizedX = width / 2 + xCoordinate
            canvas.drawCircle(normalizedX, getDotYCoordinate().toFloat(), getRadius(xCoordinate), getPaint(xCoordinate))
        }
    }

    /**
     * Set the dimensions of the IndefinitePagerIndicator.
     * Width is calculated below with getCalculatedWidth().
     * Height is simply the diameter of the largest circle.
     *
     * TODO: Add support for padding.
     * TODO: Add support for MATCH_PARENT & FILL_PARENT
     */
    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        setMeasuredDimension(getCalculatedWidth(), 2 * selectedDotRadiusPx)
    }

    /**
     * Get the x coordinate for a dot based on the position in the pager.
     */
    private fun getDotXCoordinate(pagerPosition: Int): Float =
            (pagerPosition - intermediateSelectedItemPosition) * (dotSeparationDistancePx + 2 * dotRadiusPx) + ((dotSeparationDistancePx + 2 * dotRadiusPx) * offsetPercent)

    /**
     * Get the y coordinate for a dot.
     *
     * The bottom of the view is y = 0 and a dot is drawn from the center, so therefore
     * the y coordinate is simply the radius.
     */
    private fun getDotYCoordinate(): Int {
        return selectedDotRadiusPx
    }

    /**
     * Calculates a dot's radius based on x position.
     *
     * If the x position is within 1 dot's length of x = 0, it is the currently selected dot.
     *
     * If the x position is within a threshold (half the width of the number of non fading dots),
     * it is a normal sized dot.
     *
     * If the x position is outside of the above threshold, it is a fading dot or not visible. The
     * radius is calculated based on a interpolator percentage of how far the
     * viewpager/recyclerview has scrolled.
     */
    private fun getRadius(xCoordinate: Float): Float {
        val xAbs = Math.abs(xCoordinate)
        // Get the x coordinate where dots begin showing as fading dots (x coordinates > half of width of all large dots)
        val largeDotThreshold = dotCount.toFloat() / 2 * (dotSeparationDistancePx + 2 * dotRadiusPx)
        return when {
            xAbs < (dotSeparationDistancePx + 2 * dotRadiusPx) / 2 -> selectedDotRadiusPx.toFloat()
            xAbs <= largeDotThreshold -> dotRadiusPx.toFloat()
            else -> {
                // Determine how close the dot is to the edge of the view for scaling the size of the dot
                val percentTowardsEdge = (xAbs - largeDotThreshold) / (getCalculatedWidth() / 2.01f - largeDotThreshold)
                interpolator.getInterpolation(1 - percentTowardsEdge) * dotRadiusPx
            }
        }
    }

    /**
     * Returns the dot's color based on x coordinate, similar to {@link #getRadius(Float)}.
     *
     * If the x position is within 1 dot's length of x = 0, it is the currently selected dot.
     *
     * All other dots will be the normal specified dot color.
     */
    private fun getPaint(xCoordinate: Float): Paint = when {
        Math.abs(xCoordinate) < (dotSeparationDistancePx + 2 * dotRadiusPx) / 2 -> selectedDotPaint
        else -> dotPaint
    }

    /**
     * Get the calculated with of the view.
     *
     * Calculated by the total number of visible dots (normal & fading).
     *
     * TODO: Add support for padding.
     */
    private fun getCalculatedWidth(): Int {
        val maxNumVisibleDots = dotCount + 2 * fadingDotCount
        return (maxNumVisibleDots - 1) * (dotSeparationDistancePx + 2 * dotRadiusPx) + 2 * dotRadiusPx
    }

    /**
     * Attach a RecyclerView to the Pager Indicator.
     *
     * If ViewPager previously attached, remove this class (page change listener) and set to null.
     * If other RecyclerView previously attached, remove internal scroll listener.
     */
    fun attachToRecyclerView(recyclerView: RecyclerView?) {
        viewPager?.removeOnPageChangeListener(this)
        viewPager = null

        this.recyclerView?.removeOnScrollListener(internalRecyclerScrollListener)

        this.recyclerView = recyclerView
        internalRecyclerScrollListener = InternalRecyclerScrollListener()
        this.recyclerView?.addOnScrollListener(internalRecyclerScrollListener)
    }

    /**
     * Attach a ViewPager to the Pager Indicator.
     *
     * If RecyclerView previously attached, scroll listener will be removed and RV set to null.
     * If other ViewPager previously attached, remove reference to this class (page change listener).
     */
    fun attachToViewPager(viewPager: ViewPager?) {
        recyclerView?.removeOnScrollListener(internalRecyclerScrollListener)
        recyclerView = null

        this.viewPager?.removeOnPageChangeListener(this)

        this.viewPager = viewPager
        this.viewPager?.addOnPageChangeListener(this)

        selectedItemPosition = viewPager?.currentItem!!
    }

    private fun getPagerItemCount(): Int = when {
        recyclerView != null -> recyclerView?.adapter?.itemCount!!
        viewPager != null -> viewPager?.adapter?.count!!
        else -> 0
    }

    /**
     * ViewPager.OnPageChangeListener implementation.
     *
     * Used to update the intermediateSelectedPosition & offsetPercent when the page is scrolled.
     * OffsetPercent multiplied by -1 to account for natural swipe movement.
     */
    override fun onPageScrolled(position: Int, positionOffset: Float, positionOffsetPixels: Int) {
        intermediateSelectedItemPosition = position
        offsetPercent = positionOffset * -1
        invalidate()
    }

    override fun onPageSelected(position: Int) {
        selectedItemPosition = position
        invalidate()
    }

    override fun onPageScrollStateChanged(state: Int) {
        // Not implemented
    }

    /**
     * Internal scroll listener to handle the scaling/fading/selected dot states for a RecyclerView.
     */
    internal inner class InternalRecyclerScrollListener : RecyclerView.OnScrollListener() {

        /**
         * The previous most visible child page in the RecyclerView.
         *
         * Used to differentiate between the current most visible child page to correctly determine
         * the currently selected item and percentage scrolled.
         */
        private var previousMostVisibleChild: View? = null

        /**
         * Determine based on the percentage a child viewholder's view is visible what position
         * is the currently selected.
         *
         * Use this percentage to also calculate the offsetPercentage
         * used to scale dots.
         */
        override fun onScrolled(recyclerView: RecyclerView?, dx: Int, dy: Int) {

            val view = getMostVisibleChild()
            setIntermediateSelectedItemPosition(view)
            offsetPercent = view!!.left.toFloat() / view.measuredWidth

            val layoutManager = recyclerView?.layoutManager as LinearLayoutManager
            val visibleItemPosition = if (dx >= 0) layoutManager.findLastVisibleItemPosition() else layoutManager.findFirstVisibleItemPosition()

            if (previousMostVisibleChild !== layoutManager.findViewByPosition(visibleItemPosition)) {
                selectedItemPosition = intermediateSelectedItemPosition
            }
            previousMostVisibleChild = view
            invalidate()
        }

        /**
         * Returns the currently most visible viewholder view in the Recyclerview.
         *
         * The most visible view is determined based on percentage of the view visible. This is
         * calculated below in calculatePercentVisible().
         */
        private fun getMostVisibleChild(): View? {
            var mostVisibleChild: View? = null
            var mostVisibleChildPercent = 0f
            for (i in recyclerView?.layoutManager?.childCount!! - 1 downTo 0) {
                val child = recyclerView?.layoutManager?.getChildAt(i)
                if (child != null) {
                    val percentVisible = calculatePercentVisible(child)
                    if (percentVisible >= mostVisibleChildPercent) {
                        mostVisibleChildPercent = percentVisible
                        mostVisibleChild = child
                    }
                }
            }

            return mostVisibleChild
        }

        private fun calculatePercentVisible(child: View): Float {
            val left = child.left
            val right = child.right
            val width = child.width

            return when {
                left < 0 -> right / width.toFloat()
                right > getWidth() -> (getWidth() - left) / width.toFloat()
                else -> 1f
            }
        }

        private fun setIntermediateSelectedItemPosition(mostVisibleChild: View?) {
            intermediateSelectedItemPosition = recyclerView?.findContainingViewHolder(mostVisibleChild)?.adapterPosition!!
        }
    }
}