package org.qas.qtest.api.services.project;

import org.qas.api.AuthServiceException;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.internal.model.Field;
import org.qas.qtest.api.services.project.model.*;

import java.util.List;
import java.util.concurrent.Future;

/**
 * ProjectServiceAsync
 *
 * @author Dzung Nguyen
 * @version $Id ProjectServiceAsync 2014-03-27 22:17:30z dungvnguyen $
 * @since 1.0
 */
public interface ProjectServiceAsync extends ProjectService {
  /**
   * Lists all projects in user space.
   *
   * @param listProjectRequest the given {@link ListProjectRequest request} to get projects.
   * @return the list of {@link Project} information.
   * @throws AuthServiceException if an error occurs during getting project.
   */
  Future<List<Project>> listProjectAsync(ListProjectRequest listProjectRequest) throws AuthServiceException;

  /**
   * Lists all projects in user space.
   *
   * @param listProjectRequest the given {@link ListProjectRequest request} to get projects.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the list of {@link Project} information.
   * @throws AuthServiceException if an error occurs during getting project.
   */
  Future<List<Project>> listProjectAsync(ListProjectRequest listProjectRequest,
                                           AsyncHandler<ListProjectRequest, List<Project>> asyncHandler) throws AuthServiceException;

  /**
   * Creates project in user space.
   *
   * @param createProjectRequest the given {@link CreateProjectRequest create project request}.
   * @return the created project instance.
   * @throws AuthServiceException if an error occurs during creating project.
   */
  Future<Project> createProjectAsync(CreateProjectRequest createProjectRequest) throws AuthServiceException;

  /**
   * Creates project in user space.
   *
   * @param createProjectRequest the given {@link CreateProjectRequest create project request}.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the created project instance.
   * @throws AuthServiceException if an error occurs during creating project.
   */
  Future<Project> createProjectAsync(CreateProjectRequest createProjectRequest,
                                     AsyncHandler<CreateProjectRequest, Project> asyncHandler)
    throws AuthServiceException;

  /**
   * Creates module from the given {@link CreateModuleRequest create module request}
   * object instance.
   *
   * @param createModuleRequest the given {@link CreateModuleRequest create module request} to set.
   * @return the create module instance.
   * @throws AuthServiceException if an error occurs during creating module.
   */
  Future<Module> createModuleAsync(CreateModuleRequest createModuleRequest) throws AuthServiceException;

  /**
   * Creates module from the given {@link CreateModuleRequest create module request}
   * object instance.
   *
   * @param createModuleRequest the given {@link CreateModuleRequest create module request} to set.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the create module instance.
   * @throws AuthServiceException if an error occurs during creating module.
   */
  Future<Module> createModuleAsync(CreateModuleRequest createModuleRequest,
                                   AsyncHandler<CreateModuleRequest, Module> asyncHandler)
    throws AuthServiceException;

  /**
   * Gets module from the given {@link GetModuleRequest get module request} object
   * instance.
   *
   * @param getModuleRequest the given {@link GetModuleRequest get module request}
   * @return the {@link Module module} instance.
   * @throws AuthServiceException if an error occurs during getting module.
   */
  Future<Module> getModuleAsync(GetModuleRequest getModuleRequest) throws AuthServiceException;

  /**
   * Gets module from the given {@link GetModuleRequest get module request} object
   * instance.
   *
   * @param getModuleRequest the given {@link GetModuleRequest get module request}
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the {@link Module module} instance.
   * @throws AuthServiceException if an error occurs during getting module.
   */
  Future<Module> getModuleAsync(GetModuleRequest getModuleRequest,
                                AsyncHandler<GetModuleRequest, Module> asyncHandler) throws AuthServiceException;

  /**
   * Lists modules from the given {@link ListModuleRequest get module request} object
   * instance.
   *
   * @param listModuleRequest the given {@link ListModuleRequest list module request}
   * @return the list of {@link Module module}s.
   * @throws AuthServiceException if an error occurs during listing modules.
   */
  Future<List<Module>> listModuleAsync(ListModuleRequest listModuleRequest) throws AuthServiceException;

  /**
   * Lists modules from the given {@link ListModuleRequest get module request} object
   * instance.
   *
   * @param listModuleRequest the given {@link ListModuleRequest list module request}
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the list of {@link Module module}s.
   * @throws AuthServiceException if an error occurs during listing modules.
   */
  Future<List<Module>> listModuleAsync(ListModuleRequest listModuleRequest,
                                       AsyncHandler<ListModuleRequest, List<Module>> asyncHandler)
      throws AuthServiceException;
  
  /**
   * Creates custom field from the given {@link CreateCustomFieldRequest} instance.
   *
   * @param createCustomFieldRequest the given {@link CreateCustomFieldRequest} instance to set.
   * @return the {@link Field} instance.
   * @throws AuthServiceException if an error occurs during creating custom field to project.
   */
  Future<Field> createCustomFieldAsync(CreateCustomFieldRequest createCustomFieldRequest) throws AuthServiceException;

  /**
   * Creates custom field from the given {@link CreateCustomFieldRequest} instance.
   *
   * @param createCustomFieldRequest the given {@link CreateCustomFieldRequest} instance to set.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the {@link Field} instance.
   * @throws AuthServiceException if an error occurs during creating custom field to project.
   */
  Future<Field> createCustomFieldAsync(CreateCustomFieldRequest createCustomFieldRequest,
                                       AsyncHandler<CreateCustomFieldRequest, Field> asyncHandler) throws AuthServiceException;

  /**
   * Updates system field from the given {@link UpdateSystemFieldRequest} instance.
   *
   * @param updateSystemFieldRequest the given {@link UpdateSystemFieldRequest} instance to set.
   * @throws AuthServiceException if an error occurs during
   */
  Future<Field> updateSystemFieldAsync(UpdateSystemFieldRequest updateSystemFieldRequest) throws AuthServiceException;

  /**
   * Updates system field from the given {@link UpdateSystemFieldRequest} instance.
   *
   * @param updateSystemFieldRequest the given {@link UpdateSystemFieldRequest} instance to set.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @throws AuthServiceException if an error occurs during
   */
  Future<Field> updateSystemFieldAsync(UpdateSystemFieldRequest updateSystemFieldRequest,
                                       AsyncHandler<UpdateSystemFieldRequest, Field> asyncHandler) throws AuthServiceException;

  /**
   * Gets all fields from the given field request.
   *
   * @param getFieldsRequest the given {@link GetFieldsRequest GET fields request} to get
   *                         all fields.
   * @return the list of fields.
   * @throws AuthServiceException if an error occurs during getting fields.
   */
  Future<List<Field>> getFieldsAsync(GetFieldsRequest getFieldsRequest) throws AuthServiceException;

  /**
   * Gets all fields from the given field request.
   *
   * @param getFieldsRequest the given {@link GetFieldsRequest GET fields request} to get
   *                         all fields.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the list of fields.
   * @throws AuthServiceException if an error occurs during getting fields.
   */
  Future<List<Field>> getFieldsAsync(GetFieldsRequest getFieldsRequest,
                                     AsyncHandler<GetFieldsRequest, List<Field>> asyncHandler)
      throws AuthServiceException;

  /**
   * Gets user permission.
   *
   * @param getUserPermissionsRequest the given {@link GetUserPermissionsRequest} instance.
   * @return the {@link UserPermissions} object.
   * @throws AuthServiceException if an error occurs during getting user permissions.
   */
  Future<UserPermissions> getUserPermissionsAsync(GetUserPermissionsRequest getUserPermissionsRequest)
      throws AuthServiceException;

  /**
   * Gets user permission.
   *
   * @param getUserPermissionsRequest the given {@link GetUserPermissionsRequest} instance.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the {@link UserPermissions} object.
   * @throws AuthServiceException if an error occurs during getting user permissions.
   */
  Future<UserPermissions> getUserPermissionsAsync(GetUserPermissionsRequest getUserPermissionsRequest,
                                                  AsyncHandler<GetUserPermissionsRequest, UserPermissions> asyncHandler)
      throws AuthServiceException;

  /**
   * List user permissions.
   *
   * @param listUserPermissionsRequest the given {@link ListUserPermissionsRequest} instance.
   * @return the list of {@link UserPermissions} objects.
   * @throws AuthServiceException if an error occurs during getting user permissions.
   */
  Future<List<UserPermissions>> listUserPermissionsAsync(ListUserPermissionsRequest listUserPermissionsRequest)
      throws AuthServiceException;

  /**
   * Lists user permissions.
   *
   * @param listUserPermissionsRequest the given {@link ListUserPermissionsRequest} instance.
   * @param asyncHandler callback handler for events in the life-cycle of the request.
   *                     Users could provide the implementation of the for callback
   *                     methods in this interface to process the operation result or
   *                     handle the exception.
   * @return the list of {@link UserPermissions} objects.
   * @throws AuthServiceException if an error occurs during getting user permissions.
   */
  Future<List<UserPermissions>> listUserPermissionsAsync(ListUserPermissionsRequest listUserPermissionsRequest,
                                                  AsyncHandler<ListUserPermissionsRequest, List<UserPermissions>> asyncHandler)
      throws AuthServiceException;
}
