package org.qas.qtest.api.services.host;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.qas.api.AuthClientException;
import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.services.host.model.*;

/**
 * HostServiceAsyncClient
 *
 * @author Dzung Nguyen
 * @version $Id HostServiceAsyncClient 2015-03-14 19:14:30z dzungvnguyen $
 * @since 1.0
 */
public class HostServiceAsyncClient extends HostServiceClient
    implements HostServiceAsync {
  /**
   * Constructs a new client to invoke service method on HostService using
   * the default qTest credentials provider and default client configuration options.
   */
  public HostServiceAsyncClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the default qTest credentials provider and default client configuration options.
   *
   * @param executorService the executor service for executing asynchronous request.
   */
  public HostServiceAsyncClient(ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to HostService
   */
  public HostServiceAsyncClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to HostService.
   * @param executorService the executor service for executing asynchronous request.
   */
  public HostServiceAsyncClient(ClientConfiguration clientConfiguration, ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public HostServiceAsyncClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param executorService the executor service for executing asynchronous request.
   */
  public HostServiceAsyncClient(QTestCredentials credentials, ExecutorService executorService) {
    this(credentials, new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to HostService
   */
  public HostServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    this(credentials, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to HostService
   * @param executorService the executor service for executing asynchronous request.
   */
  public HostServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration,
                                    ExecutorService executorService) {
    this(new StaticQTestCredentialsProvider(credentials), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to HostService
   */
  public HostServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                    ClientConfiguration clientConfiguration) {
    this(credentialsProvider, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on HostService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to HostService
   * @param executorService the executor service for executing asynchronous request.
   */
  public HostServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                    ClientConfiguration clientConfiguration, ExecutorService executorService) {
    super(credentialsProvider, clientConfiguration);
    this.executorService = executorService;
  }

  @Override
  public Future<AutomationAgent> registerAutomationAgentAsync(
      final CreateAutomationAgentRequest automationAgentRequest) throws AuthClientException {
    return executorService.submit(new Callable<AutomationAgent>() {
      @Override
      public AutomationAgent call() throws Exception {
        return registerAutomationAgent(automationAgentRequest);
      }
    });
  }

  @Override
  public Future<AutomationAgent> registerAutomationAgentAsync(
      final CreateAutomationAgentRequest automationAgentRequest,
      final AsyncHandler<CreateAutomationAgentRequest, AutomationAgent> asyncHandler)
      throws AuthClientException {
    return executorService.submit(new Callable<AutomationAgent>() {
      @Override
      public AutomationAgent call() throws Exception {
        AutomationAgent result;
        try {
          result = registerAutomationAgent(automationAgentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(automationAgentRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<AutomationHost> registerAutomationHostAsync(
      final CreateAutomationHostRequest automationHostRequest) throws AuthClientException {
    return executorService.submit(new Callable<AutomationHost>() {
      @Override
      public AutomationHost call() throws Exception {
        return registerAutomationHost(automationHostRequest);
      }
    });
  }

  @Override
  public Future<AutomationHost> registerAutomationHostAsync(
      final CreateAutomationHostRequest automationHostRequest,
      final AsyncHandler<CreateAutomationHostRequest, AutomationHost> asyncHandler) throws AuthClientException {
    return executorService.submit(new Callable<AutomationHost>() {
      @Override
      public AutomationHost call() throws Exception {
        AutomationHost result;

        try {
          result = registerAutomationHost(automationHostRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(automationHostRequest, result);

        return result;
      }
    });
  }
  
  @Override
  public Future<List<Job>> listJobsAsync(final ListJobRequest request) throws AuthClientException {
    return executorService.submit(new Callable<List<Job>>() {
      @Override
      public List<Job> call() throws Exception {
         return listJobs(request);
      }
    });
  }
  
  @Override
  public Future<List<Job>> listJobsAsync(final ListJobRequest request,
                                         final AsyncHandler<ListJobRequest, List<Job>> asyncHandler)
      throws AuthClientException {
    return executorService.submit(new Callable<List<Job>>() {
      @Override
      public List<Job> call() throws Exception {
        List<Job> result;
        try {
          result = listJobs(request);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(request, result);
        return result;
      }
    });
  }

  @Override
  public Future<PongMessage> pingHostAsync(final PingAutomationHostRequest pingHostRequest)
    throws AuthClientException {
    return executorService.submit(new Callable<PongMessage>() {
      @Override
      public PongMessage call() throws Exception {
        return pingHost(pingHostRequest);
      }
    });
  }

  @Override
  public Future<PongMessage> pingHostAsync(final PingAutomationHostRequest pingHostRequest,
                                           final AsyncHandler<PingAutomationHostRequest, PongMessage> asyncHandler) throws AuthClientException {
    return executorService.submit(new Callable<PongMessage>() {
      @Override
      public PongMessage call() throws Exception {
        PongMessage result;
        try {
          result = pingHost(pingHostRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(pingHostRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<AutomationAgent> updateAutomationAgentAsync(
    final UpdateAutomationAgentRequest automationAgentRequest) throws AuthClientException {
    return executorService.submit(new Callable<AutomationAgent>() {
      @Override
      public AutomationAgent call() throws Exception {
        return updateAutomationAgent(automationAgentRequest);
      }
    });
  }

  @Override
  public Future<AutomationAgent> updateAutomationAgentAsync(
    final UpdateAutomationAgentRequest automationAgentRequest,
    final AsyncHandler<UpdateAutomationAgentRequest, AutomationAgent> asyncHandler)
    throws AuthClientException {
    return executorService.submit(new Callable<AutomationAgent>() {
      @Override
      public AutomationAgent call() throws Exception {
        AutomationAgent result;
        try {
          result = updateAutomationAgent(automationAgentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(automationAgentRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Void> activateAutomationAgentAsync(final ActivateAutomationAgentRequest activateAgentRequest)
    throws AuthClientException {
    return executorService.submit(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        activateAutomationAgent(activateAgentRequest);
        return null;
      }
    });
  }

  @Override
  public Future<Void> activateAutomationAgentAsync(final ActivateAutomationAgentRequest activateAgentRequest,
                                                   final AsyncHandler<ActivateAutomationAgentRequest, Void> asyncHandler)
    throws AuthClientException {
    return executorService.submit(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        try {
          activateAutomationAgent(activateAgentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(activateAgentRequest, null);
        return null;
      }
    });
  }
  
  @Override
  public Future<Void> updateJobStatusAsync(final UpdateJobStatusRequest updateJobStatusRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        updateJobStatus(updateJobStatusRequest);
        return null;
      }
    });
  }
  
  @Override
  public Future<Void> updateJobStatusAsync(final UpdateJobStatusRequest updateJobStatusRequest,
                                            final AsyncHandler<UpdateJobStatusRequest, Void> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<Void>() {
      @Override
      public Void call() throws Exception {

        try {
          updateJobStatus(updateJobStatusRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(updateJobStatusRequest, null);
        return null;
      }
    });
  }
  
  @Override
  public Future<Void> deleteAutomationAgentAsync(final DeleteAutomationAgentRequest deleteAutomationAgentRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        deleteAutomationAgent(deleteAutomationAgentRequest);
        return null;
      }
    });
  }
  
  @Override
  public Future<Void> deleteAutomationAgentAsync(final DeleteAutomationAgentRequest deleteAutomationAgentRequest,
                                            final AsyncHandler<DeleteAutomationAgentRequest, Void> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<Void>() {
      @Override
      public Void call() throws Exception {

        try {
          deleteAutomationAgent(deleteAutomationAgentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(deleteAutomationAgentRequest, null);
        return null;
      }
    });
  }

  @Override
  public void shutdown() {
    super.shutdown();
    executorService.shutdown();
  }
}
